//
//  TabloGraphe.java
//  Petit tableur avec trac de graphes
//
// paramtre d'entre de l'applet:
//   - fichierXML: nom du fichier XML avec une balise TABLE comme racine

package tablographe;

import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.io.*;
import java.net.*;

import COM.Subrahmanyam.table.Table;
import COM.Subrahmanyam.table.TElement;
import ptolemy.plot.Plot;
import xml.*;

public class TabloGraphe extends Applet implements ActionListener {
    
    final static String texte_aide = "Pour tracer un graphe avec les valeurs d'une colonne par rapport  une autre,\n" +
        "slectionner les deux colonnes (en bas) et cliquer sur tracer, ventuellement avec l'option de connexion des points.\n" +
        "Pour calculer les valeurs d'une colonne  partir des valeurs des autres colonnes, cliquer sur la premire " +
        "cellule de cette colonne, cliquer sur la zone de texte en haut, et entrer une expression en " +
        "commenant par = et utilisant les premires cellules des autres colonnes, par exemple '=A1+B1'. "+
        "L'expression est calcule automatiquement pour toutes les lignes.\n" +
        "Chaque colonne correspond  une lettre de l'alphabet, et chaque ligne  un nombre, B3 correspond donc  la " +
        "troisime cellule de la deuxime colonne. On peut aussi utiliser les noms des colonnes, c'est  dire le titre " +
        "qui est affich en gras, par exemple en crivant '=T^2/a^3' si 'T' et 'a' sont des titres de colonnes.\n" +
        "Les oprateurs utilisables sont : + - * / ^ (valus dans l'ordre usuel).\n" +
        "Les fonctions connues sont : log ln exp sin cos tan sin acos atan abs sqrt pent pfrac moyenne.\n" +
        " La fonction pent donne la partie entire d'un nombre (l'entier plus petit le plus grand).\n" +
        " pfrac(x) donne x - pent(x) .\n" +
        " moyenne donne la moyenne des valeurs d'une colonne  partir de son nom.\n" +
        "On peut aussi utiliser des parenthses pour grouper les expressions.\n";
    Table table;
    Label nomCellule;
    TextField champ;
    Choice xchoice;
    Choice ychoice;
    Checkbox check;
    String[][] valeurs = { {"1", "2", "3"},
                           {"4", "5", "6"},
                           {"", "", ""} };
    String[] colonnes = {"colonne 1", "colonne 2", "colonne 3"};
    int selr;
    int selc;
    int numdroite;
    
    // pour l'affichage du graphe:
    int colx=0, coly=1; // colonnes slectionnes pour le graphe
    boolean connection = false;
    Label labcoord;
    TextField tffita, tffitb;
    Double[] vx;
    Double[] vy;
    Plot plot;
    Button bfit;
    Button bdroite;
    
    public void init() {
        lireTable();
        affTable();
    }
    
    public void lireTable() {
        String fname = getParameter("fichierXML");
        XMLTree tree = new XMLTree(null, "", null, "");
        URL hurle;
        try {
            hurle = new URL(getDocumentBase(), fname);
        } catch (MalformedURLException ex) {
            System.err.println("MalformedURLException: " + ex.getMessage());
            return;
        }
        Parser_XML p = new Parser_XML(hurle, System.out, tree);
        p.parse(0);
        if (!"TABLE".equalsIgnoreCase(tree.tag)) {
            System.err.println("Mauvais fichier XML !");
            return;
        }
        int maxi = 0;
        int maxj = 0;
        XMLTree tr;
        XMLTree td;
        for (tr=tree.first_child; tr!=null; tr=tr.next_brother) {
            if ("TR".equalsIgnoreCase(tr.tag)) {
                if ("TD".equalsIgnoreCase(tr.first_child.tag))
                    maxi++;
                if (maxj == 0)
                    for (td=tr.first_child; td!=null; td=td.next_brother)
                        if ("TH".equalsIgnoreCase(td.tag) || "TD".equalsIgnoreCase(td.tag))
                            maxj++;
            }
        }
        valeurs = new String[maxj][maxi];
        colonnes = new String[maxj];
        int i;
        int j;
        for (tr=tree.first_child, i=0; tr!=null; tr=tr.next_brother) {
            if ("TR".equalsIgnoreCase(tr.tag)) {
                for (td=tr.first_child, j=0; td!=null; td=td.next_brother, j++) {
                    if ("TH".equalsIgnoreCase(td.tag))
                        colonnes[j] = td.val;
                    else if ("TD".equalsIgnoreCase(td.tag))
                        valeurs[j][i] = td.val;
                }
                if ("TD".equalsIgnoreCase(tr.first_child.tag))
                    i++;
            }
        }
    }
    
    public void enregistrerTable(URL url) {
        try {
            OutputStream out = url.openConnection().getOutputStream();
            enregistrerTable(out);
        } catch (IOException ex) {
            System.err.println("enregistrerTable: " + ex.getMessage());
        }
    }
    
    public void enregistrerTable(OutputStream out) {
        XMLTree tree = new XMLTree(null, "TABLE", null, "");
        for (int i=0; i<valeurs.length; i++) {
            XMLTree tr = new XMLTree(tree, "TR", null, "");
            tree.addChild(tr);
            for (int j=0; j<valeurs[i].length; j++) {
                XMLTree td = new XMLTree(tr, "TD", null, valeurs[i][j]);
                tr.addChild(td);
            }
        }
        tree.writeTree(out, null);
    }
    
    public void affTable() {
        selr = -1;
        selc = -1;
        removeAll();
        setLayout(new BorderLayout());
        table = new Table();
        table.setSize(600,300);
        table.showHorizontalSeparator(true);
        table.showVerticalSeparator(true);
        table.setHiliteMode(Table.HILITE_ELEMENT);
        table.setClickMode(Table.DOUBLE_CLICK);
        table.setColors(Color.black, new Color((float)1.0,(float)1.0,(float)0.6),
            Color.black, Color.white,
            Color.black, new Color((float)0.6,(float)1.0,(float)0.6));
        
        for (int i=0; i<valeurs.length; i++)
            table.addColumn(colonnes[i], valeurs[i]);

        table.addItemListener(new IL());
        table.addActionListener(new ALTable());
        table.addKeyListener(new TableKeys());
        
        add(table, BorderLayout.CENTER);
        
        Panel hpane = new Panel(new FlowLayout(FlowLayout.LEFT));
        nomCellule = new Label("      ");
        hpane.add(nomCellule);
        champ = new TextField(40);
        champ.addActionListener(new ALChamp());
        hpane.add(champ);
        Button baide = new Button("aide");
        baide.addActionListener(new ALAide());
        hpane.add(baide);
        add(hpane, BorderLayout.NORTH);
        
        Panel gpane = new Panel(new GridLayout(2,1));
        Panel gpane1 = new Panel();
        Label lab = new Label("Afficher le graphe de : ");
        gpane1.add(lab);
        ychoice = new Choice();
        for (int i=0; i<valeurs.length; i++)
            ychoice.add(colonnes[i]);
        if (valeurs.length >= coly)
            ychoice.select(coly);
        gpane1.add(ychoice);
        lab = new Label(" fonction de : ");
        gpane1.add(lab);
        xchoice = new Choice();
        for (int i=0; i<valeurs.length; i++)
            xchoice.add(colonnes[i]);
        if (valeurs.length >= colx)
            xchoice.select(colx);
        gpane1.add(xchoice);
        gpane.add(gpane1);
        Panel gpane2 = new Panel();
        check = new Checkbox("Connecter les points", connection);
        gpane2.add(check);
        Button btracer = new Button("Tracer");
        btracer.addActionListener(this);
        btracer.setActionCommand("tracer");
        gpane2.add(btracer);
        gpane.add(gpane2);
        add(gpane, BorderLayout.SOUTH);
        toutRecalculer();
        validate();
    }

    // ItemListener pour la table
    class IL implements ItemListener {
        public void itemStateChanged(ItemEvent ie) {
            if (ie.getStateChange() == ItemEvent.SELECTED) {
                actionChamp();
                Object[] sel = table.getSelectedObjects();
                if (sel.length > 0) {
                    TElement el = (TElement)sel[0];
                    selr = el.getRow();
                    selc = el.getColumn();
                    // nomCellule.setText bloque plusieurs secondes avec Java 1.4.1 sur MacOS X
                    nomCellule.setText(((char)('A'+selc)) + "" + (selr+1) + " ");
                    champ.setText(valeurs[selc][selr]);
                    table.requestFocus();
                }
            }
        }
    }

    // ActionListener pour la table
    class ALTable implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            selr = table.getSelRowIndex();
            selc = table.getSelColumnIndex();
            champ.requestFocus();
        }
    }

    // ActionListener pour le champ de texte
    class ALChamp implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            actionChamp();
        }
    }
    
    // ActionListener pour l'aide
    class ALAide implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            InfoWindow f = new InfoWindow("Aide", texte_aide);
            Point apppoint = getLocationOnScreen();
            f.setLocation(apppoint.x+50, apppoint.y+50);
            f.setVisible(true);
        }
    }
    
    // appel quand le texte d'une cellule est modifi
    public void actionChamp() {
        if (selr != -1 && selc != -1) {
            String texte = champ.getText();
            if (!texte.equals(valeurs[selc][selr])) {
                valeurs[selc][selr] = texte;
                if (texte.startsWith("=")) {
                    texte = texte.substring(1).trim();
                    if (!"".equals(texte)) {
                        texte = ajParentheses(texte);
                        Parametre param = parser(texte);
                        Double d = param.evaluer();
                        if (d == null)
                            texte = "";
                        else
                            texte = joliFormattage(d.doubleValue(), 8);
                    }
                }
                table.setElement(selr, selc, texte);
                toutRecalculer();
                table.requestFocus();
                copieAuto();
            }
        }
    }
    
    // copie du contenu de la premire cellule d'une colonne vers les autres cellules
    public void copieAuto() {
        String s1 = valeurs[selc][selr];
        if (selr != 0)
            return;
        if (s1 == null || !s1.startsWith("=") || s1.indexOf("moyenne(") != -1)
            return;
        /*
        for (int i=1; i<valeurs[0].length; i++)
            if (!"".equals(valeurs[selc][i]))
                return;
        */
        // remplacement des noms des colonnes par leur codages, par ex. 'Tsid' -> 'C1'
        for (int i=1; i<s1.length(); i++) {
            char cav = s1.charAt(i-1);
            if ((cav >= 'a' && cav <='z') || (cav >= 'A' && cav <= 'Z'))
                continue;
            for (int j=0; j<colonnes.length; j++) {
                char cap = ' ';
                if (i+colonnes[j].length()<s1.length())
                    cap = s1.charAt(i+colonnes[j].length());
                if ((cap >= 'a' && cap <= 'z') || (cap >= 'A' && cap <= 'Z') || (cap >= '0' && cap <= '9'))
                    continue;
                if (colonnes[j] != null && s1.length() >= i + colonnes[j].length() &&
                        colonnes[j].equals(s1.substring(i, i+colonnes[j].length()))) {
                    s1 = s1.substring(0, i) + (char)('A'+j) + "1" + s1.substring(i+colonnes[j].length());
                    break;
                }
            }
        }
        valeurs[selc][selr] = s1;
        
        String s;
        for (int ir=1; ir<valeurs[0].length; ir++) {
            int i1, i2, ni1;
            s = s1;
            i1 = -1;
            for (int i=0; i<s.length()-1; i++) {
                if (Character.isLetter(s.charAt(i)) && Character.isDigit(s.charAt(i+1))) {
                    i1 = i;
                    break;
                }
            }
            ni1 = i1;
            while (ni1 != -1) {
                i2 = i1+1;
                while (i2 < s.length() && s.charAt(i2) >= '0' && s.charAt(i2) <= '9')
                    i2++;
                if (i2 > i1+1) {
                    int nl = -1;
                    try {
                        nl = (new Integer(s.substring(i1+1, i2))).intValue();
                    } catch (NumberFormatException ex) {
                        System.err.println("NumberFormatException: " + ex.getMessage());
                        return;
                    }
                    s = s.substring(0,i1+1) + (nl+ir) + s.substring(i2);
                }
                ni1 = i1+1;
                if (ni1 >= s.length())
                    ni1 = -1;
                else {
                    char c = s.charAt(ni1);
                    while (ni1 < s.length() && !((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
                        ni1++;
                        if (ni1 < s.length())
                            c = s.charAt(ni1);
                    }
                    if (ni1 >= s.length())
                        ni1 = -1;
                }
                i1 = ni1;
            }
            valeurs[selc][ir] = s;
            table.setElement(ir, selc, s);
        }
        toutRecalculer();
    }
    
    class TableKeys extends KeyAdapter {
        public void keyPressed(KeyEvent e) {
            int code = e.getKeyCode();
            switch (code) {
                case KeyEvent.VK_UP: move(-1, 0); break;
                case KeyEvent.VK_DOWN: move(1, 0); break;
                case KeyEvent.VK_LEFT: move(0, -1); break;
                case KeyEvent.VK_RIGHT: move(0, 1); break;
                case KeyEvent.VK_ENTER: champ.requestFocus(); break;
            }
        }
        public void move(int mr, int mc) {
            Object[] sel = table.getSelectedObjects();
            if (sel != null && sel.length > 0) {
                TElement el = (TElement)sel[0];
                int r = el.getRow();
                int c = el.getColumn();
                if (r + mr >= 0 && r + mr < valeurs[0].length &&
                    c + mc >= 0 && c + mc < valeurs.length) {
                    if (mr != 0) {
                        table.selectRow(r); // pour corriger un bug dans Table
                        table.selectRow(r + mr);
                    }
                    if (mc != 0) {
                        table.selectColumn(c); // pour corriger un bug dans Table
                        table.selectColumn(c + mc);
                    }
                    selr = r + mr;
                    selc = c + mc;
                    nomCellule.setText(((char)('A'+selc)) + "" + (selr+1) + " ");
                    champ.setText(valeurs[selc][selr]);
                }
            }
        }
    }
    
    private Double doubleOuNull(String s) {
        if (s == null)
            return(null);
        try {
            Double d = new Double(s);
            return(d);
        } catch (NumberFormatException ex) {
            return(null);
        }
    }
    
    public void affGraphe(int indx, int indy, boolean connect) {
        colx = indx;
        coly = indy;
        String xlab = colonnes[indx];
        String ylab = colonnes[indy];
        String[] svx = valeurs[indx];
        String[] svy = valeurs[indy];
        vx = new Double[svx.length];
        vy = new Double[svy.length];
        try {
            for (int i=0; i<svx.length; i++) {
                if (svx[i].startsWith("="))
                    vx[i] = calculer(svx[i]);
                else
                    vx[i] = doubleOuNull(svx[i]);
            }
            for (int i=0; i<svy.length; i++)
                if (svy[i].startsWith("="))
                    vy[i] = calculer(svy[i]);
                else
                    vy[i] = doubleOuNull(svy[i]);
        } catch (NumberFormatException ex) {
            System.err.println("NumberFormatException: " + ex.getMessage());
        }
        
        removeAll();
        plot = new Plot();
        add(plot, BorderLayout.CENTER);
        plot.setMarksStyle("cross");
        plot.setConnected(connect);
        plot.setXLabel(xlab);
        plot.setYLabel(ylab);
        boolean sansNombreX = true;
        if (vx.length < 2)
            sansNombreX = false;
        for (int i=0; i<vx.length; i++)
            if (vx[i] != null)
                sansNombreX = false;
        if (sansNombreX) {
            for (int i=0; i<vx.length; i++) {
                vx[i] = new Double(i+1);
                plot.addXTick(svx[i], i+1);
            }
        }
        for (int i=0; i<vx.length; i++)
            if (vx[i] != null && vy[i] != null)
                plot.addPoint(0, vx[i].doubleValue(), vy[i].doubleValue(), true);
        plot.addMouseMotionListener(new MML());
        
        Panel pbas = new Panel(new GridLayout(2, 1));
        
        labcoord = new Label();
        pbas.add(labcoord);
        
        Panel pfit = new Panel(new FlowLayout());
        bfit = new Button("Ajustement");
        bfit.addActionListener(this);
        bfit.setActionCommand("fit");
        pfit.add(bfit);
        pfit.add(new Label("fit: a="));
        tffita = new TextField(6);
        pfit.add(tffita);
        pfit.add(new Label(" b="));
        tffitb = new TextField(6);
        pfit.add(tffitb);
        bdroite = new Button("Tracer");
        bdroite.addActionListener(this);
        bdroite.setActionCommand("droite");
        numdroite = 1;
        pfit.add(bdroite);
        Button btableau = new Button("Revenir au tableau");
        btableau.addActionListener(this);
        btableau.setActionCommand("tableau");
        pfit.add(btableau);
        pbas.add(pfit);
        
        add(pbas, BorderLayout.SOUTH);

        plot.setVisible(true);
        validate();
    }
    
    class MML implements MouseMotionListener {
        public void mouseDragged(MouseEvent e) {
        }
        public void mouseMoved(MouseEvent e) {
            double coords[] = plot.getPointCoordinates(e.getX(), e.getY());
            labcoord.setText("x="+joliFormattage(coords[0], 3) + "  y="+joliFormattage(coords[1], 3));
            validate();
        }
    }
    
    public String joliFormattage(double d, int precision) {
        double facteur_precision = Math.pow(10, precision);
        int exposant = (int)Math.floor(Math.log(Math.abs(d))/Math.log(10));
        double mantisse = d / (Math.pow(10,exposant));
        String s;
        if (exposant == 0) {
            mantisse = Math.round(mantisse*facteur_precision)/facteur_precision;
            s = ""+mantisse;
        } else if (exposant == 1) {
            mantisse = Math.round(mantisse*facteur_precision)/(facteur_precision/10);
            s = ""+mantisse;
        } else if (exposant == -1) {
            String signe;
            if (mantisse < 0)
                signe = "-";
            else
                signe = "";
            mantisse = Math.abs(Math.round(mantisse*facteur_precision));
            s = signe + "0."+(int)mantisse;
        } else {
            mantisse = Math.round(mantisse*facteur_precision)/facteur_precision;
            s = mantisse + "E" + exposant;
        }
        return(s);
    }
    
    public void tracerDroite() {
        String sa = tffita.getText();
        String sb = tffitb.getText();
        Double da = doubleOuNull(sa);
        Double db = doubleOuNull(sb);
        if (da != null && db != null) {
            numdroite++;
            plot.setMarksStyle("none",numdroite);
            plot.setConnected(true);
            double a = da.doubleValue();
            double b = db.doubleValue();
            plot.setMarksStyle("none",1);
            plot.setConnected(true);
            for (int i=0; i<vx.length; i++)
                if (vx[i] != null) {
                    double x = vx[i].doubleValue();
                    plot.addPoint(numdroite, x, a*x + b, true);
                }
        }
    }
    
    public void doFit() {
        int n=0;
        for (int i=0; i<vx.length; i++)
            if (vx[i] != null && vy[i] != null)
                n++;
        double[] tx = new double[n];
        double[] ty = new double[n];
        int j=0;
        for (int i=0; i<vx.length; i++)
            if (vx[i] != null && vy[i] != null) {
                tx[j] = vx[i].doubleValue();
                ty[j] = vy[i].doubleValue();
                j++;
            }
        double[] tc = fit(tx, ty);
        double y;
        plot.setMarksStyle("none",1);
        plot.setConnected(true);
        for (int i=0; i<tx.length; i++) {
            y = tc[0]*tx[i] + tc[1];
            plot.addPoint(1, tx[i], y, true);
        }
        //System.out.println("a="+tc[0]+" b="+tc[1]);
        tffita.setText(joliFormattage(tc[0], 3));
        tffitb.setText(joliFormattage(tc[1], 3));
        bfit.setEnabled(false);
        validate();
    }
    
    // fit droite moindre carrs
    public double[] fit(double[] tx, double[] ty) {
        int n = tx.length;
        double sx = 0;
        for (int i=0; i<n; i++)
            sx += tx[i];
        double sy = 0;
        for (int i=0; i<n; i++)
            sy += ty[i];
        double sx2 = 0;
        for (int i=0; i<n; i++)
            sx2 += tx[i] * tx[i];
        double sy2 = 0;
        for (int i=0; i<n; i++)
            sy2 += ty[i] * ty[i];
        double sxy = 0;
        for (int i=0; i<n; i++)
            sxy += tx[i] * ty[i];
        double[] tc = new double[2];
        tc[0] = (n*sxy - sx*sy)/(n*sx2 - sx*sx);
        tc[1] = (sy*sx2 - sx*sxy)/(n*sx2 - sx*sx);
        return(tc);
    }
    
    public void actionPerformed(ActionEvent ae) {
        String cmd = ae.getActionCommand();
        if ("tableau".equals(cmd))
            affTable();
        else if ("tracer".equals(cmd)) {
            int indx = xchoice.getSelectedIndex();
            int indy = ychoice.getSelectedIndex();
            connection = check.getState();
            affGraphe(indx, indy, connection);
        } else if ("fit".equals(cmd))
            doFit();
        else if ("droite".equals(cmd))
            tracerDroite();
    }
    
    
    // fonctions de tableur
    
    public void toutRecalculer() {
        for (int i=0; i<valeurs.length; i++)
            for (int j=0; j<valeurs[i].length; j++) {
                String texte = valeurs[i][j];
                if (texte.startsWith("=")) {
                    Double d = calculer(texte);
                    if (d == null)
                        texte = "";
                    else
                        texte = joliFormattage(d.doubleValue(), 8);
                    table.setElement(j, i, texte);
                }
            }
    }
    
    public Double calculer(String s) {
        s = s.substring(1).trim();
        if ("".equals(s))
            return(null);
        s = ajParentheses(s);
        Parametre param = parser(s);
        Double d = param.evaluer();
        return(d);
    }
    
    public String valeurCellule(String nom) {
        if (nom.length() < 2)
            return null;
        nom = nom.toUpperCase();
        char c = nom.charAt(0);
        int nc = c - 'A' + 1;
        int nl = -1;
        try {
            nl = (new Integer(nom.substring(1))).intValue();
        } catch (NumberFormatException ex) {
            System.err.println("NumberFormatException: " + ex.getMessage());
            return(null);
        }
        if (nc < 1 || nc > valeurs.length || nl < 1 || nl > valeurs[0].length)
            return(null);
        return(valeurs[nc-1][nl-1]);
    }
    
    public String ajParentheses(String s) {
        String sops = "^/*-+";
        for (int iops=0; iops<sops.length(); iops++) {
            char cops = sops.charAt(iops);
            int indop = s.indexOf(cops);
            if (indop > 0 && "^*/-+Ee".indexOf(s.charAt(indop-1)) != -1)
                indop = -1; // un - ou + aprs un E ou un oprateur n'est pas un oprateur !
            int nindop = indop;
            int im,ip;
            char cm=' ',cp=' ';
            int pp;
            boolean ajp;
            while (nindop != -1) {
                ajp = false;
                im = indop - 1;
                if (im >= 0)
                    cm = s.charAt(im);
                pp = 0;
                while (im >= 0 && (pp != 0 || cm != '(') &&
                        (pp != 0 || (sops.indexOf(cm) == -1 ||
                        (im > 0 && "^*/-+Ee".indexOf(s.charAt(im-1)) != -1)))) {
                    if (cm == ')')
                        pp++;
                    else if (cm == '(')
                        pp--;
                    im--;
                    if (im >= 0)
                        cm = s.charAt(im);
                }
                if (im < 0 || sops.indexOf(cm) != -1)
                    ajp = true;
                ip = indop + 1;
                if (ip >= 0)
                    cp = s.charAt(ip);
                pp = 0;
                while (ip < s.length() && (pp != 0 || cp != ')') &&
                        (pp != 0 || (sops.indexOf(cp) == -1 ||
                        (ip > 0 && "^*/-+Ee".indexOf(s.charAt(ip-1)) != -1)))) {
                    if (cp == '(')
                        pp++;
                    else if (cp == ')')
                        pp--;
                    ip++;
                    if (ip < s.length())
                        cp = s.charAt(ip);
                }
                if (ip >= s.length() || sops.indexOf(cp) != -1)
                    ajp = true;
                if (ajp) {
                    s = s.substring(0, im+1) + "(" + s.substring(im+1, ip) + ")" +
                        s.substring(ip);
                    indop++;
                }
                nindop = s.substring(indop+1).indexOf(cops);
                indop = nindop + indop+1;
            }
        }
        return(s);
    }
    
    public Parametre parser(String s) {
        String sops = "^/*-+";
        if (s.charAt(0) == '(' && s.charAt(s.length()-1) == ')') {
            boolean retparok = true;
            // on vrifie si on peut retirer les parenthses qui entourent l'expression
            int pp = 0;
            for (int i=1; i<s.length()-1; i++) {
                if (s.charAt(i) == '(')
                    pp++;
                else if (s.charAt(i) == ')')
                    pp--;
                if (pp < 0) {
                    retparok = false;
                    break;
                }
            }
            if (retparok)
                s = s.substring(1, s.length()-1);
        }
        int indop = -1;
        int pp = 0;
        for (int i=0; i<s.length(); i++) {
            if (pp == 0 && sops.indexOf(s.charAt(i)) != -1 &&
                    (i==0 || "^*/-+Ee".indexOf(s.charAt(i-1)) == -1)) {
                indop = i;
                break;
            } else if (s.charAt(i) == '(')
                pp++;
            else if (s.charAt(i) == ')')
                pp--;
        }
        if (indop == -1) {
            boolean nb = true;
            for (int i=0; i<s.length(); i++)
                if ("0123456789.,-+ ".indexOf(s.charAt(i)) == -1  &&
                        ((s.charAt(i) != 'E' && s.charAt(i) != 'e') ||
                        (i == 0 || "0123456789".indexOf(s.charAt(i-1)) == -1))) {
                    nb = false;
                    break;
                }
            if (nb) {
                double d;
                try {
                    d = (new Double(s)).doubleValue();
                } catch (NumberFormatException ex) {
                    d = 0;
                }
                return(new Nombre(d));
            } else {
                int indf = s.indexOf('(');
                if (indf != -1 && s.charAt(s.length()-1) == ')') {
                    Fonction fct = new Fonction(s.substring(0,indf));
                    int indv = 0;
                    Parametre p1 = null;
                    Parametre p2 = null;
                    s = s.substring(indf+1, s.length()-1);
                    indv = s.indexOf(',');
                    if (indv != -1) {
                        p1 = parser(s.substring(0,indv).trim());
                        s = s.substring(indv+1);
                        p2 = parser(s.trim());
                    } else
                        p1 = parser(s.trim());
                    return(new Expression(fct, p1, p2));
                } else
                    return(new Variable(s));
            }
        } else {
            Fonction op = new Fonction(s.charAt(indop));
            String s1 = s.substring(0,indop).trim();
            if ("".equals(s1))
                s1 = "0"; // -(2+3) = 0 - (2+3)
            Parametre p1 = parser(s1);
            String s2 = s.substring(indop+1).trim();
            Parametre p2 = parser(s2);
            return(new Expression(op, p1, p2));
        }
    }
    
    class Expression implements Parametre {
        Fonction fct;
        Parametre p1,p2;
        
        public Expression(Fonction fct, Parametre p1, Parametre p2) {
            this.fct = fct;
            this.p1 = p1;
            this.p2 = p2;
        }
        public Double evaluer() {
            if (fct.type == Fonction.moyenne) {
                if (!(p1 instanceof Variable))
                    return(null);
                String nomCol = ((Variable)p1).nom;
                int numc = -1;
                if (nomCol.length() == 1) {
                    char c = Character.toUpperCase(nomCol.charAt(0));
                    if (c >= 'A' && c <= 'Z')
                        numc = c - 'A';
                }
                if (numc == -1) {
                    for (int i=0; i<colonnes.length; i++)
                        if (nomCol.equals(colonnes[i]))
                            numc = i;
                }
                if (numc == -1)
                    return(null);
                return(new Double(moyenne(numc)));
            } else if (p2 == null) {
                Double d1 = p1.evaluer();
                if (d1 == null)
                    return(null);
                return(fct.evaluer(d1.doubleValue(), 0));
            } else {
                Double d1 = p1.evaluer();
                Double d2 = p2.evaluer();
                if (d1 == null || d2 == null)
                    return(null);
                return(fct.evaluer(d1.doubleValue(), d2.doubleValue()));
            }
        }
    }
    
    public double moyenne(int nc) {
        double somme = 0;
        int nb = 0;
        for (int i=0; i<valeurs[nc].length; i++) {
            String s = valeurs[nc][i];
            if (s != null && s.startsWith("="))
                s = table.getElement(i, nc);
            Double d = doubleOuNull(s);
            if (d != null) {
                somme += d.doubleValue();
                nb++;
            }
        }
        return(somme / nb);
    }
    
    interface Parametre {
        public abstract Double evaluer();
    }
    
    class Variable implements Parametre {
        String nom;
        
        public Variable(String nom) {
            this.nom = nom;
        }
        public Double evaluer() {
            if (nom == null)
                return(null);
            if ("e".equals(nom))
                return(new Double(Math.E));
            else if ("pi".equals(nom))
                return(new Double(Math.PI));
            String vc = valeurCellule(nom);
            if (vc == null)
                for (int i=0; i<colonnes.length; i++)
                    if (nom.equals(colonnes[i])) {
                        vc = valeurCellule(((char)('A'+i))+"1");
                        break;
                    }
            if (vc == null)
                return(null);
            if (vc.startsWith("=")) {
                nom = nom.toUpperCase();
                char c = nom.charAt(0);
                int nc = c - 'A';
                int nl = -1;
                try {
                    nl = (new Integer(nom.substring(1))).intValue() - 1;
                } catch (NumberFormatException ex) {
                    System.err.println("NumberFormatException: " + ex.getMessage());
                    return(null);
                }
                vc = table.getElement(nl, nc);
                if (vc == null)
                    return(null);
            }
            try {
                Double d = new Double(vc);
                return(d);
            } catch (NumberFormatException ex) {
                System.err.println("NumberFormatException: " + ex.getMessage());
                return(null);
            }
        }
    }
    
    static class Nombre implements Parametre {
        double valeur;
        
        public Nombre(double valeur) {
            this.valeur = valeur;
        }
        public Double evaluer() {
            return(new Double(valeur));
        }
    }
    
    class Fonction {
        int type;
        static final int addition = 1;
        static final int soustraction = 2;
        static final int multiplication = 3;
        static final int division = 4;
        static final int puissance = 5;
        static final int logdecimal = 6;
        static final int ln = 7;
        static final int exp = 8;
        static final int sin = 9;
        static final int cos = 10;
        static final int tan = 11;
        static final int asin = 12;
        static final int acos = 13;
        static final int atan = 14;
        static final int abs = 15;
        static final int sqrt = 16;
        static final int partie_entiere = 17;
        static final int partie_frac = 18;
        static final int moyenne = 19;
        
        // constructeur pour un oprateur
        public Fonction(char c) {
            switch (c) {
                case '+': type = addition; break;
                case '-': type = soustraction; break;
                case '*': type = multiplication; break;
                case '/': type = division; break;
                case '^': type = puissance; break;
                default: type = 0; break;
            }
        }
        
        // constructeur pour une fonction
        public Fonction(String nom) {
            if ("log".equals(nom))
                type = logdecimal;
            else if ("ln".equals(nom))
                type = ln;
            else if ("exp".equals(nom))
                type = exp;
            else if ("sin".equals(nom))
                type = sin;
            else if ("cos".equals(nom))
                type = cos;
            else if ("tan".equals(nom))
                type = tan;
            else if ("asin".equals(nom))
                type = asin;
            else if ("acos".equals(nom))
                type = acos;
            else if ("atan".equals(nom))
                type = atan;
            else if ("abs".equals(nom))
                type = abs;
            else if ("sqrt".equals(nom))
                type = sqrt;
            else if ("pent".equals(nom))
                type = partie_entiere;
            else if ("pfrac".equals(nom))
                type = partie_frac;
            else if ("moyenne".equals(nom))
                type = moyenne;
            else
                type = 0;
        }
        
        public Double evaluer(double v1, double v2) {
            double d;
            switch (type) {
                case addition: d = v1 + v2; break;
                case soustraction:  d = v1 - v2; break;
                case multiplication:  d = v1 * v2; break;
                case division:  d = v1 / v2; break;
                case puissance:  d = Math.pow(v1, v2); break;
                case logdecimal:  d = Math.log(v1)/Math.log(10); break;
                case ln:  d = Math.log(v1); break;
                case exp:  d = Math.exp(v1); break;
                case sin:  d = Math.sin(v1); break;
                case cos:  d = Math.cos(v1); break;
                case tan:  d = Math.tan(v1); break;
                case asin:  d = Math.asin(v1); break;
                case acos:  d = Math.acos(v1); break;
                case atan:  d = Math.atan(v1); break;
                case abs:  d = Math.abs(v1); break;
                case sqrt:  d = Math.sqrt(v1); break;
                case partie_entiere:  d = Math.floor(v1); break;
                case partie_frac:  d = v1 - Math.floor(v1); break;
                // moyenne: voir Expression.evaluer
                default: return(null);
            }
            return(new Double(d));
        }
    }
    
    class InfoWindow extends Frame implements ActionListener {
        public InfoWindow(String title, String message) {
            setTitle(title);
            setLayout(new BorderLayout());
            int nblignes = 5;
            if (message.length() > 200)
                nblignes = 10;
            TextArea text = new TextArea(message, nblignes, 60, TextArea.SCROLLBARS_VERTICAL_ONLY);
            text.setEditable(false);
            add(BorderLayout.CENTER, text);
            Button bok = new Button("OK");
            bok.addActionListener(this);
            Panel panel = new Panel();
            panel.add(bok);
            add(BorderLayout.SOUTH, panel);
            pack();
            text.setCaretPosition(0);
            addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    setVisible(false);
                    dispose();
                }
            });
        }
        public void actionPerformed(ActionEvent event) {
            setVisible(false);
            dispose();
        }
    }
}
