/*

Copyright (C) 2012  Observatoire de Paris

The JavaScript code in this page is free software: you can
redistribute it and/or modify it under the terms of the GNU
General Public License (GNU GPL) as published by the Free Software
Foundation, either version 3 of the License, or (at your option)
any later version.  The code is distributed WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU GPL for more details.

As additional permission under GNU GPL version 3 section 7, you
may distribute non-source (e.g., minimized or compacted) forms of
that code without the copy of the GNU GPL normally required by
section 4, provided you include this license notice and a URL
through which recipients can access the Corresponding Source.

*/

if (!window.Node) {
    var Node = {
        ELEMENT_NODE                :  1,
        ATTRIBUTE_NODE              :  2,
        TEXT_NODE                   :  3,
        CDATA_SECTION_NODE          :  4,
        ENTITY_REFERENCE_NODE       :  5,
        ENTITY_NODE                 :  6,
        PROCESSING_INSTRUCTION_NODE :  7,
        COMMENT_NODE                :  8,
        DOCUMENT_NODE               :  9,
        DOCUMENT_TYPE_NODE          : 10,
        DOCUMENT_FRAGMENT_NODE      : 11,
        NOTATION_NODE               : 12
    };
}

/**
 * Application créant une interface pour un problème donné dans une classe dérivée.
 * Une classe dérivée de SimuApplet doit définir le problème physique (paramètres et
 * calcul) avec les méthodes définies dans SimuParams.
 * Cette classe dérivée doit implémenter les méthodes
 * initCalculLive, calculLive et/ou calcul, et peut utiliser les méthodes
 * addParamIn, addParamOut et addAffichage définies dans SimuApplet.
 */
function SimuApplet(calculs) {
    // méthodes de calculs (avec les méthodes calcul et/ou initCalculLive + calculLive)
    this.calculs = calculs;
    
    // attributs de SimuParams ("publics")
    this.inparams = new Array();
    this.outparams = new Array();
    this.affichages = new Array();
    this.ensembles = new ListeValeurs();
    this.titreEnsembles = 'Ensembles';
    
    // attributs "privés"
    this.champs = null; // Array
    this.noaff = -1; // int
    this.za = null; // TypeAffichage
    this.chxens = null;
    this.selens = null; // string
    this.nomFichierXML = null; // string
    this.xmlDoc = null;
    this.delai = 50;
    this.reinit = false; // utilisé pour réinitialiser après une erreur
    
    var moi = this;
    this.actionPerformedProxy = function(e) { moi.actionPerformed(e) };
    this.itemStateChangedProxy = function(e) { moi.itemStateChanged(e) };
    
    if (window.addEventListener)
        window.addEventListener('load', function() { moi.initParams(); }, false);
    else if (window.attachEvent)
        window.attachEvent('onload', function() { moi.initParams(); });
    
    return(this);
}

// *************************************
// méthodes pour SimuParams

/** renvoit la liste des paramètres d'entrée (Array of ParamIn) */
SimuApplet.prototype.getInParams = function() {
    return(this.inparams);
}

/** renvoit la liste des paramètres de sortie (Array of ParamOut) */
SimuApplet.prototype.getOutParams = function() {
    return(this.outparams);
}

/** renvoit la liste des possibilités d'affichage (Array of Affichage) */
SimuApplet.prototype.getAffichages = function() {
    return(this.affichages);
}

/** renvoit l'affichage avec le label donné */
SimuApplet.prototype.getAffichage = function(label) {
    var i, j;
    if (label == null)
        return(null);
    for (i=0; i<this.affichages.length; i++)
        if (label == this.affichages[i].label)
            return(this.affichages[i]);
    for (i=0; i<this.affichages.length; i++)
        if (this.affichages[i].sousAff != null)
            for (j=0; j<this.affichages[i].sousAff.length; j++)
                if (label == this.affichages[i].sousAff[j].label)
                    return(this.affichages[i].sousAff[j]);
    return(null);
}

/** renvoit l'affichage choisi par l'utilisateur */
SimuApplet.prototype.getAffichageChoisi = function() {
    if (this.noaff == -1)
        return(null);
    return(this.affichages[this.noaff]);
}

/** ajoute un paramètre d'entrée à la liste */
SimuApplet.prototype.addParamIn = function(p) {
    this.inparams.push(p);
}

/** ajoute un paramètre de sortie à la liste */
SimuApplet.prototype.addParamOut = function(p) {
    this.outparams.push(p);
}

/** ajoute une option d'affichage à la liste */
SimuApplet.prototype.addAffichage = function(p) {
    this.affichages.push(p);
}

/** ajoute un ensemble de valeurs */
//public void addEnsemble(String nom, ListeValeurs l) {
SimuApplet.prototype.addEnsemble = function(nom, l) {
    this.ensembles.ajouter(nom, l);
}

/** Définit le titre des ensembles de valeurs */
SimuApplet.prototype.setTitreEnsembles = function(titre) {
    this.titreEnsembles = titre;
}

/** Spécifie le fichier XML avec la définition des paramètres et des affichages. */
SimuApplet.prototype.appletDoc = function(nomFichier) {
    this.nomFichierXML = nomFichier;
}


SimuApplet.prototype.toutEffacer = function() {
    var inputp, div_affichage;
    inputp = document.getElementById('inputp');
    if (inputp != null)
        inputp.parentNode.removeChild(inputp);
    div_affichage = document.getElementById('affichage');
    if (div_affichage != null)
        div_affichage.parentNode.removeChild(div_affichage);
}

/**
 * Affichage de la fenêtre et appel la lecture du fichier de configuration
 */
SimuApplet.prototype.initParams = function() {
    if (this.inparams.length == 0)
        this.lireParametres();
    this.noaff = -1;
    this.za = null;
    this.toutEffacer();
    var inputp = this.makeInputPanel(null);
    if (inputp != null)
        document.body.appendChild(inputp);
}

/**
 * Création de la zone avec les paramètres d'entrée
 */
SimuApplet.prototype.makeInputPanel = function(lin) {
    var inp, inputp, nbgrid, table, tbody, cellules, ip, tr, td, defauts, texte, noms, io, nom, option, par, bhelp, titre, vdef, tf, choix, ic, curseur, canvas_curseur, labcurseur, chk, actionsp, vaff, ia, aff, bok;
    
    inp = this.getInParams();
    inputp = document.createElement('table');
    inputp.setAttribute('id', 'inputp');
    //inputp.setAttribute('class', 'parametres');
    // pour IE7:
    //inputp.className = 'parametres';
    // IE7 nécessite aussi l'utilisation de tbody, et display: inline-block ne marche pas pour obtenir la taille de la zone,
    // du coup on est obligé d'utiliser une table pour tout, ce qui complique l'ajout d'éléments...
    tbody = document.createElement('tbody');
    inputp.appendChild(tbody);
    this.champs = new Array();
    try {
        defauts = null;
        if (this.ensembles.taille() > 0) {
            tr = document.createElement('tr');
            tbody.appendChild(tr);
            td = document.createElement('td');
            tr.appendChild(td);
            texte = document.createTextNode(this.titreEnsembles + ':');
            td.appendChild(texte);
            this.chxens = document.createElement('select');
            noms = this.ensembles.noms();
            for (io=0; io<noms.length; io++) {
                nom = noms[io];
                option = document.createElement('option');
                option.setAttribute('value', nom);
                option.appendChild(document.createTextNode(nom));
                this.chxens.appendChild(option);
                if (defauts == null)
                    defauts = this.ensembles.lire(nom);
                if (this.selens == nom) {
                    option.selected = true;
                    defauts = this.ensembles.lire(this.selens);
                }
            }
            if (this.chxens.addEventListener)
                this.chxens.addEventListener('change', this.itemStateChangedProxy, false);
            else
                this.chxens.attachEvent('onchange', this.itemStateChangedProxy);
            td.appendChild(this.chxens);
        }
        for (ip=0; ip<inp.length; ip++) {
            tr = document.createElement('tr');
            tbody.appendChild(tr);
            td = document.createElement('td');
            tr.appendChild(td);
            par = inp[ip];
            
            bhelp = null;
            if (this.verifExistenceAide(par)) {
                bhelp = document.createElement('input');
                bhelp.setAttribute('type', 'button');
                bhelp.setAttribute('value', Messages.get('?'));
                bhelp.setAttribute('id', 'help' + ip);
                if (bhelp.addEventListener)
                    bhelp.addEventListener('click', this.actionPerformedProxy, false);
                else
                    bhelp.attachEvent('onclick', this.actionPerformedProxy);
            }

            titre = par.titre;
            if (titre == null)
                titre = par.label;
            if (par.unit != null)
                titre += ' (' + par.unit + ')';
            if (bhelp != null)
                titre = ' ' + titre;
            
            vdef = null;
            if (lin != null)
                vdef = lin.lireString(par.label);
            else if (defauts != null) {
                try {
                    vdef = defauts.lireString(par.label);
                } catch (ex) {
                }
            }
            if (vdef == null && par.defaut != null)
                vdef = par.defaut;
            
            if (par.acquisition == 'champ') {
                if (bhelp != null)
                    td.appendChild(bhelp);
                td.appendChild(document.createTextNode(titre + ': '));
                tf = document.createElement('input');
                tf.setAttribute('type', 'text');
                tf.setAttribute('size', '10');
                if (vdef != null)
                    tf.value = vdef;
                this.champs.push(tf);
                td.appendChild(tf);
            } else if (par.acquisition == 'choix') {
                if (bhelp != null)
                    td.appendChild(bhelp);
                td.appendChild(document.createTextNode(titre + ': '));
                choix = document.createElement('select');
                for (ic=0; ic < par.choix.length; ic++) {
                    option = document.createElement('option');
                    option.setAttribute('value', par.choix[ic]);
                    option.appendChild(document.createTextNode(par.choix[ic]));
                    if (vdef == par.choix[ic])
                        option.selected = true;
                    choix.appendChild(option);
                }
                this.champs.push(choix);
                td.appendChild(choix);
            } else if (par.acquisition == 'curseur') {
                if (bhelp != null) {
                    td.appendChild(bhelp);
                    td.appendChild(document.createTextNode(titre + ': '));
                } else {
                    td.appendChild(document.createTextNode(titre + ': '));
                }
                td.appendChild(document.createElement('br'));
                canvas_curseur = document.createElement('canvas');
                canvas_curseur.setAttribute('id', 'curseur' + ip);
                td.appendChild(canvas_curseur);
                curseur = new Curseur('curseur' + ip, 0, 100, vdef == null ? 0 : vdef, 101, false, function(e) {
                    var curs = e.target;
                    var labcurseur = document.getElementById('labcurseur' + ip);
                    labcurseur.innerHTML = ' = ' + curs.getValeur();
                });
                labcurseur = document.createElement('span');
                if (vdef != null) {
                    labcurseur.innerHTML = ' = ' + vdef + '  ';
                } else {
                    labcurseur.innerHTML = ' = 0  ';
                }
                td.appendChild(labcurseur);
                this.champs.push(curseur);
            } else if (par.acquisition == 'case') {
                if (bhelp != null)
                    td.appendChild(bhelp);
                chk = document.createElement('input');
                chk.setAttribute('type', 'checkbox');
                td.appendChild(document.createTextNode(titre));
                if (vdef != null)
                    if (vdef == 'on')
                        chk.checked = true;
                td.appendChild(chk);
                this.champs.push(chk);
            } else
                this.champs.push(null);
        }
    } catch (ex) {
        this.handleException(ex);
        return(null);
    }
    tr = document.createElement('tr');
    td = document.createElement('td');
    tr.appendChild(td);
    //td.setAttribute('id', 'actions');
    vaff = this.getAffichages();
    if (this.noaff == -1) {
        for (ia=0; ia < vaff.length; ia++) {
            aff = vaff[ia];
            bok = document.createElement('input');
            bok.setAttribute('type', 'button');
            bok.setAttribute('value', aff.titre);
            bok.setAttribute('id', 'OK' + ia);
            if (bok.addEventListener)
                bok.addEventListener('click', this.actionPerformedProxy, false);
            else
                bok.attachEvent('onclick', this.actionPerformedProxy);
            td.appendChild(bok);
        }
    } else {
        aff = vaff[this.noaff];
        bok = document.createElement('input');
        bok.setAttribute('type', 'button');
        bok.setAttribute('value', Messages.get('Relancer'));
        bok.setAttribute('id', 'Relancer');
        if (bok.addEventListener)
            bok.addEventListener('click', this.actionPerformedProxy, false);
        else
            bok.attachEvent('onclick', this.actionPerformedProxy);
        td.appendChild(bok);
    }
    tbody.appendChild(tr);
    return(inputp);
}

// renvoie le premier enfant de premier niveau de l'élément XML parent avec le nom donné, ou null s'il n'est pas trouvé
SimuApplet.prototype.enfantParNom = function(parent, nom) {
    var n;
    for (n=parent.firstChild; n != null; n = n.nextSibling) {
        if (n.nodeType == Node.ELEMENT_NODE && n.nodeName == nom)
            return(n);
    }
    return(null);
}

/**
* Lit les paramètres spécifiés dans le fichier XML
*/
SimuApplet.prototype.lireParametres = function() {
    var appletdoc, listeparams, node, param_in, attribute_value, node_valeurs, tab_choix, node_choix, node_description, param_out, ensemble, n;
    
    // lecture du fichier XML si nécessaire
    if (this.xmlDoc == null)
        this.lireXML();
    
    appletdoc = null;
    if (this.xmlDoc.documentElement.nodeName == 'APPLET_SIMULAB')
        appletdoc = this.xmlDoc.documentElement;
    else // cas DEADOC
        appletdoc = this.enfantParNom(this.xmlDoc.documentElement, 'APPLETDOC');
    if (appletdoc == null) {
        alert(Messages.get('sans_appletdoc'));
        return;
    }
    listeparams = this.enfantParNom(appletdoc, 'LISTEPARAMS');
    if (listeparams == null) {
        alert(Messages.get('sans_listeparams'));
        return;
    }
    
    for (node = listeparams.firstChild; node != null; node = node.nextSibling) {
        
        // Paramètre pour la compatibilité ascendente
        // (PARAMETRE a été remplacé par PARAMETRE_ENTREE)
        if ( node.nodeName == 'PARAMETRE' ) {
            
            param_in = new ParamIn();
            attribute_value = null;

            if ((attribute_value = node.getAttribute('acquisition')) != null)
                param_in.acquisition = attribute_value;

            if ((attribute_value = node.getAttribute('defaut')) != null)
                param_in.defaut = attribute_value;
                
            if ((attribute_value = node.getAttribute('label')) != null)
                param_in.label = attribute_value;

            if ((attribute_value = node.getAttribute('titre')) != null)
                param_in.titre = attribute_value;

            if ((attribute_value = node.getAttribute('type')) != null)
                param_in.type = attribute_value;

            if ((attribute_value = node.getAttribute('unite')) != null)
                param_in.unit = attribute_value;
            
            if (node.firstChild != null && node.firstChild.nodeValue != '')
                param_in.description = node.firstChild.nodeValue;
            
            this.addParamIn(param_in);
        }
        
       // Paramètres d'entrée
       if ( node.nodeName == 'PARAMETRE_ENTREE' ) {
            param_in = new ParamIn();
            attribute_value = null;
            
            if (  ( attribute_value = node.getAttribute('acquisition') ) != null ) {
                param_in.acquisition = attribute_value;
                
                if ( attribute_value == 'choix' ) {
                    node_valeurs = this.enfantParNom(node, 'VALEURS');
                    if (node_valeurs == null) {
                        alert('Erreur: acquisition choix sans élément VALEURS sous PARAMETRE_ENTREE');
                    } else {
                        tab_choix = new Array();
                        
                        for (node_choix=node_valeurs.firstChild; node_choix != null; node_choix = node_choix.nextSibling) {
                            if (node_choix.nodeType == Node.ELEMENT_NODE && node_choix.nodeName == 'CHOIX') {
                                if ( (attribute_value = node_choix.getAttribute('valeur')) != null )
                                    tab_choix.push(attribute_value);
                                else
                                    tab_choix.push('');
                            }
                        }
                        param_in.choix = tab_choix;
                    }
                }
            }

            if (  (attribute_value = node.getAttribute('defaut')) != null )
                param_in.defaut = attribute_value;
            
            if (  (attribute_value = node.getAttribute('label')) != null )
                param_in.label = attribute_value;

            if (  (attribute_value = node.getAttribute('titre')) != null )
                param_in.titre = attribute_value;

            if (  (attribute_value = node.getAttribute('type')) != null )
                param_in.type = attribute_value;

            if (  (attribute_value = node.getAttribute('unite')) != null )
                param_in.unit = attribute_value;
            
            node_description = this.enfantParNom(node, 'DESCRIPTION');
            if ( node_description != null && node_description.firstChild != null )
                param_in.description = node_description.firstChild.nodeValue;
            
            this.addParamIn(param_in);
        }
        
        // Paramètres de sortie
        if ( node.nodeName == 'PARAMETRE_SORTIE' ) {
            param_out = new ParamOut();
            attribute_value = null;

            if (  (attribute_value = node.getAttribute('label')) != null )
                param_out.label = attribute_value;
            
            if (  (attribute_value = node.getAttribute('type')) != null )
                 param_out.type = attribute_value;
            
            if (  (attribute_value = node.getAttribute('titre')) != null )
                 param_out.titre = attribute_value;
            
            if (  (attribute_value = node.getAttribute('unite')) != null )
                param_out.unit = attribute_value;
            
            this.addParamOut(param_out);
        }
        
        // Paramètres d'affichage
        if ( node.nodeName == 'AFFICHAGE' )
            this.addAffichage(this.lireAffichage(node));
       
       // Ensembles de valeurs
       if ( node.nodeName == 'ENSEMBLE' ) {
            ensemble = new ListeValeurs();
            
            for (n = node.firstChild; n != null; n = n.nextSibling)
                if (n.nodeType == Node.ELEMENT_NODE && n.nodeName == 'VALEURPARAM')
                    ensemble.ajouterString(n.getAttribute('label'), n.getAttribute('valeur'));
            
            this.addEnsemble(node.getAttribute('titre'), ensemble);
       }
    } 
}

/** Parcours récursif des paramètres d'affichages */
//public Affichage lireAffichage(XMLTree noeud_affichage) {
SimuApplet.prototype.lireAffichage = function(noeud_affichage) {
    var affichage, sous_affichages, n;
    
    affichage = new Affichage();

    this.parcourirParametreAffichage(noeud_affichage, affichage);
    
    this.lireListeRefParams(noeud_affichage, affichage);
    
    sous_affichages = new Array(); // de Affichage 
    
    for (n = noeud_affichage.firstChild; n != null; n = n.nextSibling)
        if (n.nodeName == 'AFFICHAGE')
            sous_affichages.push(this.lireAffichage(n));
    
    if (sous_affichages.length > 0)
        affichage.sousAff = sous_affichages;
    
    return affichage; 
}

//public void lireListeRefParams(XMLTree noeud_affichage, Affichage affichage) {
SimuApplet.prototype.lireListeRefParams = function(noeud_affichage, affichage) {
    var listes, nbrefs, n, i, n2;
    
    listes = new Array(); // listes de références, Array de Array de string
    // toutes les listes de références doivent avoir le même nombre de références
    nbrefs = 0;
    for (n = noeud_affichage.firstChild; n != null; n = n.nextSibling)
        if (n.nodeName == 'LISTEREFPARAMS') {
            liste = new Array(); // liste de références
            for (n2 = n.firstChild; n2 != null; n2 = n2.nextSibling)
                if (n2.nodeName == 'REFPARAM')
                    liste.push(n2.getAttribute('nom'));
            listes.push(liste);
            if (nbrefs == 0)
                nbrefs = liste.length;
            else if (nbrefs != liste.length)
                alert('Erreur: pas le même nombre de références dans toutes les listes de références de paramètres');
        }
    if (listes.length > 0)
        affichage.params = listes;
}

/** Sert à lire les attributs d'un paramètre d'affichage */
//public void parcourirParametreAffichage(XMLTree noeud_affichage, Affichage affichage) {
SimuApplet.prototype.parcourirParametreAffichage = function(noeud_affichage, affichage) {
    var attribute_value = null;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('type') ) != null )
        affichage.type = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('titre') ) != null )
        affichage.titre = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('label') ) != null )
        affichage.label = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('xdebut') ) != null )
        affichage.xdebut = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('xfin') ) != null )
        affichage.xfin = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('ydebut') ) != null )
        affichage.ydebut = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('yfin') ) != null )
        affichage.yfin = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('xlog') ) != null )
        affichage.xlog = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('ylog') ) != null )
       affichage.ylog = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('fond') ) != null )
        affichage.fond = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('xdebutfond') ) != null )
        affichage.xdebutfond = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('xfinfond') ) != null )
        affichage.xfinfond = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('ydebutfond') ) != null )
        affichage.ydebutfond = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('yfinfond') ) != null )
        affichage.yfinfond = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('zooming') ) != null )
        affichage.zooming = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('imgdimx') ) != null )
        affichage.imgdimx = parseInt(attribute_value);
    
    if (  ( attribute_value = noeud_affichage.getAttribute('imgdimy') ) != null )
        affichage.imgdimy = parseInt(attribute_value);
    
    // palette ?
    
    if (  ( attribute_value = noeud_affichage.getAttribute('connecter') ) != null )
        affichage.connecter = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('ordreSousAff') ) != null )
// attention aux majuscules pour ordreSousAff (remplacées par le parser dans une ancienne version)
        affichage.ordreSousAff = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('boucle') ) != null )
        affichage.boucle = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('points') ) != null )
        affichage.points = attribute_value;
    
    if (  ( attribute_value = noeud_affichage.getAttribute('couleurs') ) != null )
        affichage.couleurs = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('histogramme') ) != null )
        affichage.histogramme = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('effacer') ) != null )
        affichage.effacer = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('ajuster') ) != null )
        affichage.ajuster = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('recadrer') ) != null )
        affichage.recadrer = (attribute_value == 'true');
    
    if (  ( attribute_value = noeud_affichage.getAttribute('carre') ) != null )
        affichage.carre = (attribute_value == 'true');  
}

/**
 * Traitement des actions de l'utilisateur
 */
SimuApplet.prototype.actionPerformed = function(e) {
    var cmd, noparam;
    
    cmd = (e.target || e.srcElement).getAttribute('id');
    if (cmd.indexOf('OK') == 0) {
        this.noaff = parseInt(cmd.substring(2));
        this.startSimu(true);
    } else if (cmd == 'Reset') {
        if (this.za != null && this.za.estLive())
            this.za.stop();
        this.initParams();
    } else if (cmd == 'Relancer') {
        if (this.za != null && this.za.estLive())
            this.za.stop();
        this.startSimu(this.reinit);
    } else if (cmd.indexOf('help') == 0) {
        noparam = parseInt(cmd.substring(4));
        this.aideParam(noparam);
    }
}

//public void changerValeurs(ListeValeurs l) throws SimuException {
SimuApplet.prototype.changerValeurs = function(l) {
    var inp, i, par, val, chp;
    
    inp = this.getInParams();
    for (i=0; i<inp.length; i++) {
        par = inp[i];
        try {
            val = l.lireString(par.label);
        } catch (ex) {
            continue;
        }
        chp = this.champs[i];
        if (chp.nodeName.toLowerCase() == 'input' && chp.getAttribute('type') == 'text')
            chp.value = val;
        else if (chp instanceof Curseur)
            chp.setValeur(val);
        else if (chp.nodeName.toLowerCase() == 'select')
            chp.value = val;
        else if (chp.nodeName.toLowerCase() == 'input' && chp.getAttribute('type') == 'checkbox')
            chp.value = (val == 'on');
    }
}

SimuApplet.prototype.itemStateChanged = function(e) {
    // itemStateChanged est appelé par chxens via itemStateChangedProxy
    var nom, l;
    
    nom = this.chxens.value;
    try {
        l = this.ensembles.lire(nom);
        this.changerValeurs(l);
    } catch (ex) {
        this.handleException(ex);
        return;
    }
}

/**
 * Traitement d'une erreur
 */
SimuApplet.prototype.handleException = function(ex) {
    var s;
    if (ex.message)
        s = ex.message;
    else
        s = ex;
    if (ex.fileName)
        s += ' in ' + ex.fileName;
    if (ex.lineNumber)
        s += ' line ' + ex.lineNumber;
    alert(Messages.get('Erreur') + ': ' + s);
}

/**
 * Lecture des champs des paramètres d'entrée
 */
//public ListeValeurs lireChamps() throws SimuException {
SimuApplet.prototype.lireChamps = function() {
    var lin, inp, i, par, text, d;
    
    lin = new ListeValeurs();
    inp = this.getInParams();
    for (i=0; i<inp.length; i++) {
        par = inp[i];
        text = null;
        if (this.champs[i].nodeName.toLowerCase() == 'input' && this.champs[i].getAttribute('type') == 'text') {
            text = this.champs[i].value;
            // test si le type de text est par.type
            if (par.type == 'nombre') {
                d = parseFloat(text);
                if (d == NaN)
                    throw new Error(text + ' ' + Messages.get('pas_un_nombre'));
            }
        } else if (this.champs[i].nodeName.toLowerCase() == 'select') {
            text = this.champs[i].value;
        } else if (this.champs[i] instanceof Curseur) {
            text = toString(this.champs[i].getValeur());
        } else if (this.champs[i].nodeName.toLowerCase() == 'input' && this.champs[i].getAttribute('type') == 'checkbox') {
            if (this.champs[i].checked)
                text = 'on';
            else
                text = 'off';
        }
        lin.ajouterString(par.label, text);
    }
    if (this.chxens != null)
        this.selens = this.chxens.value;
    return(lin);
}

/**
 * vérification que les paramètres de l'affichage sont tous déclarés comme ParamOut
 */
//protected void verifParametres(Affichage aff, Vector outp) {
SimuApplet.prototype.verifParametres = function(aff, outp) {
    var ip, jp, nomp, trouv, io, ia;
    
    if (aff.params != null)
        for (ip=0; ip<aff.params.length; ip++)
            for (jp=0; jp<aff.params[ip].length; jp++) {
                nomp = aff.params[ip][jp];
                trouv = false;
                for (io=0; io<outp.length; io++)
                    if (nomp != null && nomp == outp[io].label)
                        trouv = true;
                if (!trouv)
                    alert(Messages.get('parametre_affichage') + ' ' + nomp +
                        ' ' + Messages.get('pas_declare'));
            }
    if (aff.sousAff != null)
        for (ia=0; ia<aff.sousAff.length; ia++)
            this.verifParametres(aff.sousAff[ia], outp);
}

SimuApplet.prototype.getWindowWidth = function() {
    if (window.innerWidth)
        return(window.innerWidth);
    if (window.document.documentElement.clientWidth)
        return(window.document.documentElement.clientWidth);
    return(window.document.body.clientWidth);
}

SimuApplet.prototype.getWindowHeight = function() {
    if (window.innerHeight)
        return(window.innerHeight); // dimension parfois folle avec iOS
    if (window.document.documentElement.clientHeight)
        return(window.document.documentElement.clientHeight); // IE7 retire la taille de la scrollbar
    return(window.document.body.clientHeight);
}

/**
 * Lancement de la simulation
 */
//public void startSimu(boolean init) {
SimuApplet.prototype.startSimu = function(init) {
    var lin, outp, vaff, aff, pval1, urgent, w, h, dimensions, breset;
    
    this.reinit = false;
    try {
        lin = this.lireChamps();
    } catch (ex) {
        this.handleException(ex);
        return;
    }
    outp = this.getOutParams();
    vaff = this.getAffichages();
    aff = vaff[this.noaff];
    
    if (init) {
        this.verifParametres(aff, outp);
        this.toutEffacer();
        pval1 = this.makeInputPanel(lin);
        if (pval1 == null)
            return;
        document.body.appendChild(pval1);
        if (pval1.offsetWidth > this.getWindowWidth() / 2)
            pval1.style.width = (this.getWindowWidth() / 2) + 'px';
    }
    
    try {
        if (aff == null)
            throw new Error(Messages.get('affichage_non_defini'));
        if (init || this.za == null)
            this.za = this.creationTypeAffichage(aff, false);
        urgent = null;
        if (this.za.estLive()) {
            urgent = new Chronos(this, this.za, aff.boucle, this.delai);
            this.za.setChronos(urgent);
            urgent.initCalculLive(lin);
        }
        if (init) {
            w = this.getWindowWidth() - pval1.offsetWidth - 10;
            if (w < 100)
                w = 100;
            h = this.getWindowHeight() - 10;
            if (h < 100)
                h = 100;
            dimensions = {'width': w, 'height': h};
            document.body.appendChild(this.za.initPanel(outp, dimensions));
        }
        this.za.affiche(lin, outp);
    } catch (ex) {
        this.reinit = true;
        this.handleException(ex);
        return;
    }
    
    if (init) {
        breset = document.createElement('input');
        breset.setAttribute('type', 'button');
        breset.setAttribute('value', Messages.get('Reinitialiser'));
        breset.setAttribute('id', 'Reset');
        if (breset.addEventListener)
            breset.addEventListener('click', this.actionPerformedProxy, false);
        else
            breset.attachEvent('onclick', this.actionPerformedProxy);
        pval1.firstChild.lastChild.lastChild.appendChild(breset); // tbody > tr > td
    }
}

//public TypeAffichage creationTypeAffichage(Affichage aff, boolean sousaff) throws SimuException {
SimuApplet.prototype.creationTypeAffichage = function(aff, sousaff) {
    var ta;
    if (aff.type == 'plotlive')
        ta = new AffPlotLive(this, aff, sousaff);
    else if (aff.type == 'plot')
        ta = new AffPlot(this, aff, sousaff);
    else if (aff.type == 'tableau')
        ta = new AffPlotTab(this, aff, sousaff);
    else if (aff.type == 'image')
        ta = new AffImage(this, aff, sousaff, true);
    else if (aff.type == 'imagestatique')
        ta = new AffImage(this, aff, sousaff, false);
    else if (aff.type == 'multiplot')
        ta = new AffMulti(this, aff, sousaff);
    else
        throw new Error(Messages.get('affichage_inconnu') + ': ' + aff.type);
    return(ta);
}

/**
 * titres des axes
 */
//public static String[] titresAxes(Affichage aff, Vector outp) {
SimuApplet.prototype.titresAxes = function(aff, outp) {
    var sx, sy, lsx, lsy, i, par, res;
    
    if (aff == null)
        return null;
    if (aff.params == null)
        return null;
    sx = null;
    sy = null;
    lsx = aff.params[0][0];
    lsy = aff.params[0][1];
    sx = lsx;
    sy = lsy;
    for (i=0; i<outp.length; i++) {
        par = outp[i];
        if (lsx == par.label) {
            if (par.titre != null)
                sx = par.titre;
            if (par.unit != null)
                sx += ' (' + par.unit + ')';
        }
        if (lsy == par.label) {
            if (par.titre != null)
                sy = par.titre;
            if (par.unit != null)
                sy += ' (' + par.unit + ')';
        }
    }
    res = new Array(sx, sy);
    return(res);
}

/**
 * Renvoit l'aide pour le paramètre indiqué à partir du fichier XML,
 * ou null s'il n'y a pas d'aide pour ce paramètre
 */
//protected String lireAideParam(ParamIn param) {
SimuApplet.prototype.lireAideParam = function(param) {
    var aide, appletdoc, listeparams, node, description;
    
    if (param.description != null)
        return(param.description);
    
    if (this.xmlDoc == null)
        return(null);
    
    aide = null;
    appletdoc = null;
    
    if (this.xmlDoc.documentElement.nodeName == 'APPLET_SIMULAB')
        appletdoc = this.xmlDoc.documentElement;
    else
        appletdoc = this.enfantParNom(this.xmlDoc.documentElement, 'APPLETDOC');
    if (appletdoc == null)
        alert(Messages.get('sans_appletdoc'));
    else {
        listeparams = this.enfantParNom(appletdoc, 'LISTEPARAMS');
        if (listeparams == null)
            alert(Messages.get('sans_listeparams'));
        else {
            for (node=listeparams.firstChild; node!=null; node=node.nextSibling) {
                if (node.nodeName == 'PARAMETRE') {
                    if (node.getAttribute('label') == param.label)
                        aide = node.firstChild.nodeValue;
                } else if (node.nodeName == 'PARAMETRE_ENTREE') {
                    if (node.getAttribute('label') == param.label) {
                        description = this.enfantParNom(node, 'DESCRIPTION');
                        if (description != null && description.firstChild != null)
                            aide = description.firstChild.nodeValue;
                    }
                }
            }
        }
    }
    if (aide != null)
        param.description = aide;
    return(aide);
}

/**
 * Affiche l'aide pour le paramètre indiqué à partir du fichier XML.
 * On suppose ici que le fichier XML est correct (le test est fait ailleurs).
 */
//public void aideParam(int nop) {
SimuApplet.prototype.aideParam = function(nop) {
    var par, aide, label, f;
    
    if (this.xmlDoc == null)
        this.lireXML();
    par = this.getInParams()[nop];
    aide = this.lireAideParam(par);
    label = par.label;
    if (aide == null)
        alert(Messages.get('sans_parametre') + ' ' + label + ' ' + Messages.get('dans_xml'));
    if (aide != null) {
        f = new InfoWindow(Messages.get('parametre') + ' ' + label, aide);
        f.afficher();
    }
}

/**
* Vérifie l'existence de l'aide dans le ficher de configuration XML
* @return true si l'aide existe, false sinon
*/
//protected boolean verifExistenceAide(ParamIn param) {
SimuApplet.prototype.verifExistenceAide = function(param) {
    var aide;
    
    aide = this.lireAideParam(param);
    return(aide != null && aide != '');
}

/**
 * pour bloquer le programme pendant nms millisecondes
 */
//public void dormir(long nms) {
SimuApplet.prototype.dormir = function(nms) {
    // sans gestion des threads, Javascript est incapable de faire une pause sans bloquer le navigateur
    // il vaut mieux utiliser setTimeout en Javascript
    alert("Erreur: dormir n'est pas utilisable en javascript, utiliser setDelaiIterations à la place");
}

/**
 * Durée minimum d'une itération en ms. Une pause sera effectuée si un calcul se termine plus vite.
 */
SimuApplet.prototype.setDelaiIterations = function(nms) {
    this.delai = nms;
}

SimuApplet.prototype.lireXML = function() {
    var xmlhttp, arbre, racine;
    
    xmlhttp = new XMLHttpRequest();
    xmlhttp.open('GET', this.nomFichierXML, false);
    try {
        xmlhttp.send();
    } catch (ex) {
        alert('Lecture du fichier ' + this.nomFichierXML + ': ' + (ex.message ? ex.message : ex));
        return(null);
    }
    arbre = xmlhttp.responseXML;
    racine = arbre.documentElement;
    if (racine.nodeName != 'DEADOC' && racine.nodeName != 'APPLET_SIMULAB') {
        alert(Messages.get('mauvais_xml'));
        return;
    }
    this.xmlDoc = arbre;
}

//public void initCalculLive(ListeValeurs in) throws SimuException {
SimuApplet.prototype.initCalculLive = function(lin) {
    if (this.calculs.initCalculLive != undefined)
        this.calculs.initCalculLive(lin);
}

//public ListeValeurs calculLive() throws SimuException {
SimuApplet.prototype.calculLive = function() {
    if (this.calculs.calculLive == undefined)
        throw new Error(Messages.get('calculLive_necessaire'));
    return(this.calculs.calculLive());
}

//public ListeValeurs calcul(ListeValeurs in) throws SimuException {
SimuApplet.prototype.calcul = function(lin) {
    if (this.calculs.calcul == undefined)
        throw new Error(Messages.get('calcul_necessaire'));
    return(this.calculs.calcul(lin));
}


function InfoWindow(titre, message) {
    this.titre = titre;
    this.message = message;
}
InfoWindow.prototype.afficher = function() {
    alert(this.titre + ': ' + this.message);
}
