/*
SimuApplet

Copyright (C) 2002 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package simu;

import java.awt.*;
import java.applet.*;
import java.io.*;
import java.util.Vector;
import java.awt.event.*;
import java.net.*;

import ptolemy.plot.*;
import COM.Subrahmanyam.table.*;
import xml.*;

/**
 * Applet crant une interface pour un problme donn dans une classe drive.
 * Une classe drive de SimuApplet doit dfinir le problme physique (paramtres et
 * calcul) avec les mthodes dfinies dans SimuParams.
 * Cette classe drive doit implmenter les mthodes
 * initCalculLive, calculLive et/ou calcul, et peut utiliser les mthodes
 * addParamIn, addParamOut et addAffichage dfinies dans SimuApplet.
 */
public abstract class SimuApplet extends Applet implements ActionListener, ItemListener,
    SimuParams {

    SimuParams params;
    Vector champs;
    int noaff;
    TypeAffichage za;
    Choice chxens = null;
    String selens = null;
    String nomFichierXML = null;
    XMLTree xmlDoc = null;
    ScrollPane scrollp = null;

    /**
     * Cette mthode dfini la sous-classe de SimuParams contenant l'algorithme de
     * la simulation.
     */
    public void setParams() {
        params = this;
    }
    
    /**
     * Initialisation de l'applet
     */
    public void init() {
        super.init();
        initParams();
    }
    
    /**
     * Cration de la zone avec les paramtres d'entre
     */
    public Panel makeInputPanel(ListeValeurs in) {
        Vector inp = params.getInParams();
        // inp should not be null
        Panel inputp = new Panel();
        int nbgrid = inp.size()+1;
        if (ensembles.taille() > 0)
            nbgrid++;
        inputp.setLayout(new GridLayout(nbgrid, 1));
        champs = new Vector();
        try {
            ListeValeurs defauts = null;
            if (ensembles.taille() > 0) {
                Panel p = new Panel();
                p.setLayout(new FlowLayout(FlowLayout.LEFT));
                p.add(new Label(titreEnsembles + ":"));
                chxens = new Choice();
                Vector noms = ensembles.noms();
                for (int i=0; i<noms.size(); i++) {
                    String nom = (String)noms.elementAt(i);
                    chxens.add(nom);
                    if (defauts == null)
                        defauts = (ListeValeurs)ensembles.lire(nom);
                }
                if (selens != null) {
                    chxens.select(selens);
                    defauts = (ListeValeurs)ensembles.lire(selens);
                }
                chxens.addItemListener(this);
                p.add(chxens);
                inputp.add(p);
            }
            for (int i=0; i<inp.size(); i++) {
                ParamIn par = (ParamIn)inp.elementAt(i);
                Panel p = new Panel();
                
                Button bhelp = null;
                
                if ( verifExistenceAide(par) ) {
                    bhelp = new Button(Messages.get("?"));
                    bhelp.addActionListener(this);
                    bhelp.setActionCommand("help" + i);
                }

                String titre = par.titre;
                if (titre == null)
                    titre = par.label;
                if (par.unit != null)
                    titre += " (" + par.unit + ")";
                
                String vdef = null;
                if (in != null)
                    vdef = in.lireString(par.label);
                else if (defauts != null) {
                    try {
                        vdef = defauts.lireString(par.label);
                    } catch (SimuException ex) {
                    }
                }
                if (vdef == null && par.defaut != null)
                    vdef = par.defaut;
                
                if ("champ".equals(par.acquisition)) {
                    p.setLayout(new FlowLayout(FlowLayout.LEFT));
                    if (bhelp != null)
                        p.add(bhelp);
                    p.add(new Label(titre + ":"));
                    TextField tf = new TextField(10);
                    if (vdef != null)
                        tf.setText(vdef);
                    champs.addElement(tf);
                    p.add(tf);
                } else if ("choix".equals(par.acquisition)) {
                    p.setLayout(new FlowLayout(FlowLayout.LEFT));
                    if (bhelp != null)
                        p.add(bhelp);
                    p.add(new Label(titre + ":"));
                    Choice choix = new Choice();
                    for (int j=0; j<par.choix.length; j++)
                        choix.add(par.choix[j]);
                    if (vdef != null)
                        choix.select(vdef);
                    champs.addElement(choix);
                    p.add(choix);
                } else if ("curseur".equals(par.acquisition)) {
                    GridBagLayout gridbag = new GridBagLayout();
                    GridBagConstraints c = new GridBagConstraints();
                    c.anchor = GridBagConstraints.NORTHWEST;
                    c.gridwidth = GridBagConstraints.REMAINDER;
                    c.weightx = 1.0;
                    c.weighty = 1.0;
                    p.setLayout(gridbag);
                    if (bhelp != null) {
                        Panel p2 = new Panel();
                        p2.add(bhelp);
                        p2.add(new Label(titre + ":"));
                        gridbag.setConstraints(p2, c);
                        p.add(p2);
                    } else {
                        Label lab = new Label(titre + ":");
                        gridbag.setConstraints(lab, c);
                        p.add(lab);
                    }
                    Panel psb = new Panel();
                    psb.setLayout(new BorderLayout());
                    Canvas dummyc = new Canvas();
                    dummyc.setSize(250,1);
                    psb.add(BorderLayout.NORTH, dummyc);
                    // il y a un bug dans la JVM d'Apple, on devrait mettre 100 au lieu de 110
                    Scrollbar curseur = new Scrollbar(Scrollbar.HORIZONTAL, 0, 10, 0, 110);
                    final Label labcurseur = new Label();
                    if (vdef != null) {
                        curseur.setValue((new Double(vdef)).intValue());
                        labcurseur.setText(" = " + vdef + "  ");
                    } else {
                        curseur.setValue(0);
                        labcurseur.setText(" = 0   ");
                    }
                    curseur.addAdjustmentListener(new AdjustmentListener() {
                        public void adjustmentValueChanged(AdjustmentEvent Adevt) {
                            labcurseur.setText(" = " + Integer.toString(Adevt.getValue()));
                        }
                    });
                    psb.add(BorderLayout.EAST, labcurseur);
                    psb.add(BorderLayout.CENTER, curseur);
                    champs.addElement(curseur);
                    gridbag.setConstraints(psb, c);
                    p.add(psb);
                } else if ("case".equals(par.acquisition)) {
                    p.setLayout(new FlowLayout(FlowLayout.LEFT));
                    if (bhelp != null)
                        p.add(bhelp);
                    Checkbox chk = new Checkbox(titre);
                    if (vdef != null)
                        if ("on".equals(vdef))
                            chk.setState(true);
                    p.add(chk);
                    champs.addElement(chk);
                } else
                    champs.addElement(null);
                inputp.add(p);
            }
        } catch (SimuException ex) {
            handleException(ex);
            return(null);
        }
        Panel p = new Panel();
        p.setLayout(new FlowLayout(FlowLayout.LEFT));
        Vector vaff = params.getAffichage();
        if (noaff == -1) {
            for (int i=0; i<vaff.size(); i++) {
                Affichage a = (Affichage)vaff.elementAt(i);
                Button bok = new Button(a.titre);
                bok.setActionCommand("OK" + i);
                bok.addActionListener(this);
                p.add(bok);
            }
        } else {
            Affichage a = (Affichage)vaff.elementAt(noaff);
            Button bok = new Button(Messages.get("Relancer"));
            bok.setActionCommand("Relancer");
            bok.addActionListener(this);
            p.add(bok);
        }
        inputp.add(p);
        
        if (inp.size() > 15) {
            // avec scrollpane:
            scrollp = new ScrollPane(); // dimension ajuste plus tard
            scrollp.add(inputp);
            Panel inputp2 = new Panel();
            inputp2.setLayout(new BorderLayout(0, 0));
            inputp2.add(BorderLayout.CENTER, scrollp);
            return(inputp2);
        } else {
            // sans scrollpane:
            scrollp = null;
            Panel inputp1 = new Panel();
            inputp1.setLayout(new BorderLayout(0, 0));
            inputp1.add(BorderLayout.NORTH, inputp);
            return(inputp1);
        }
    }
    
    /**
     * Affichage de la fentre et appel la lecture du fichier de configuration
     */
    public void initParams() {
        // si les paramtres sont remplis alors, l'appel s'effectue depuis une applet ancienne
        // (les paramtres sont spcifis dans le programme Java, sauf l'aide en ligne)
        if (!inparams.isEmpty() && xmlDoc == null) {
            if (nomFichierXML != null)
                lireXML(); // pour l'aide
        } else if (xmlDoc == null)
        // dans ce cas, il s'agit d'une nouvelle requrant un fichier XML
            lireParametres();

        removeAll();
        setParams();
        setLayout(new BorderLayout(0, 0));
        noaff = -1;
        za = null;
        Panel inputp = makeInputPanel(null);
        if (inputp != null)
            add(BorderLayout.NORTH, inputp);
        if (scrollp != null) {
            scrollp.setSize(scrollp.getComponent(0).getPreferredSize().width + scrollp.getVScrollbarWidth(),
                getSize().height);
            validate();
            scrollp.invalidate();
        }
        validate();
    }
    
    /**
    * Lit les paramtres spcifis dans le fichier XML
    */
    public void lireParametres() {
        
        // lecture du fichier XML si ncessaire
        if (xmlDoc == null)
            lireXML();
        
        XMLTree appletdoc = null;
        if ("APPLET_SIMULAB".equals(xmlDoc.tag))
            appletdoc = xmlDoc;
        else
            appletdoc = xmlDoc.getNode("APPLETDOC");
        if (appletdoc == null)
            System.err.println(Messages.get("sans_appletdoc"));
        else {
            
            XMLTree listeparams = appletdoc.getNode("LISTEPARAMS");
            if (listeparams == null)
                System.err.println(Messages.get("sans_listeparams"));
            else {
                
                XMLTree node;
                for (node = listeparams.first_child; node != null; node = node.next_brother) {
                    
                    // Paramtre pour la compatibilit ascendente
                    // (PARAMETRE a t remplac par PARAMETRE_ENTREE)
                    if ( node.tag.equals("PARAMETRE") ) {
                    
                        ParamIn param_in = new ParamIn();
                        String attribute_value = null;
     
                        if ((attribute_value = node.getAttVal("acquisition")) != null)
                            param_in.acquisition = attribute_value;

                        if ((attribute_value = node.getAttVal("defaut")) != null)
                            param_in.defaut = attribute_value;
                            
                        if ((attribute_value = node.getAttVal("label")) != null)
                            param_in.label = attribute_value;

                        if ((attribute_value = node.getAttVal("titre")) != null)
                            param_in.titre = attribute_value;

                        if ((attribute_value = node.getAttVal("type")) != null)
                            param_in.type = attribute_value;

                        if ((attribute_value = node.getAttVal("unite")) != null)
                            param_in.unit = attribute_value;
                        
                        if (node.val != null && !"".equals(node.val))
                            param_in.description = node.val;
                        
                        addParamIn(param_in);
                    }
                    
                   // Paramtres d'entre
                   if ( node.tag.equals("PARAMETRE_ENTREE") ) {
                        ParamIn param_in = new ParamIn();
                        String attribute_value = null;
                        
                        if (  ( attribute_value = node.getAttVal("acquisition") ) != null ) {
                            param_in.acquisition = attribute_value;
                            
                            if ( attribute_value.equals("choix") ) {
                                XMLTree node_valeurs = node.getNode("VALEURS");
                                if (node_valeurs == null) {
                                    System.err.println("Erreur: acquisition choix sans lment VALEURS sous PARAMETRE_ENTREE");
                                } else {
                                    String[] tab_choix = new String[node_valeurs.getChildrenCount()];
                                    
                                    XMLTree node_choix;
                                    for (int i = 0;  i < node_valeurs.getChildrenCount(); i++ ) {
                                        node_choix = node_valeurs.getChildAt(i);
                                        
                                        if ( (attribute_value = node_choix.getAttVal("valeur")) != null )
                                            tab_choix[i] = attribute_value;
                                        else
                                            tab_choix[i] = "";
                                    }
                                    param_in.choix = tab_choix;
                                }
                            }
                        }

                        if (  (attribute_value = node.getAttVal("defaut")) != null )
                            param_in.defaut = attribute_value;
                        
                        if (  (attribute_value = node.getAttVal("label")) != null )
                            param_in.label = attribute_value;

                        if (  (attribute_value = node.getAttVal("titre")) != null )
                            param_in.titre = attribute_value;

                        if (  (attribute_value = node.getAttVal("type")) != null )
                            param_in.type = attribute_value;

                        if (  (attribute_value = node.getAttVal("unite")) != null )
                            param_in.unit = attribute_value;
                        
                        if ( node.getNode("DESCRIPTION") != null )
                            param_in.description = node.getNode("DESCRIPTION").val;
                        
                        addParamIn(param_in);
                    }
                    
                    // Paramtres de sortie
                    if ( node.tag.equals("PARAMETRE_SORTIE") ) {
                        ParamOut param_out = new ParamOut();
                        String attribute_value = null;

                        if (  (attribute_value = node.getAttVal("label")) != null )
                            param_out.label = attribute_value;
                        
                        if (  (attribute_value = node.getAttVal("type")) != null )
                             param_out.type = attribute_value;
                        
                        if (  (attribute_value = node.getAttVal("titre")) != null )
                             param_out.titre = attribute_value;
                        
                        if (  (attribute_value = node.getAttVal("unite")) != null )
                            param_out.unit = attribute_value;
                        
                        addParamOut(param_out);
                    }
                    
                    // Paramtres d'affichage
                    if ( node.tag.equals("AFFICHAGE") )
                        addAffichage(lireAffichage(node));
                   
                   // Ensembles de valeurs
                   if ( node.tag.equals("ENSEMBLE") ) {
                        ListeValeurs ensemble = new ListeValeurs();
                        
                        for (XMLTree n = node.first_child; n != null; n = n.next_brother)
                            ensemble.ajouterString(n.getAttVal("label"), n.getAttVal("valeur"));
                        
                        addEnsemble(node.getAttVal("titre"), ensemble);
                   }
                } 
            }
        }
    }
    
    /** Parcours rcursif des paramtres d'affichages */
    public Affichage lireAffichage(XMLTree noeud_affichage) {
        Affichage affichage = new Affichage();

        parcourirParametreAffichage(noeud_affichage, affichage);
        
        lireListeRefParams(noeud_affichage, affichage);
        
        Vector sous_affichages = new Vector(); // de Affichage 
        
        for (XMLTree n = noeud_affichage.first_child; n != null; n = n.next_brother)
            if ("AFFICHAGE".equals(n.tag))
                sous_affichages.addElement(lireAffichage(n));
        
        if (sous_affichages.size() > 0) {
            affichage.sousAff = new Affichage[sous_affichages.size()];
            sous_affichages.copyInto(affichage.sousAff);
        }
         
        return affichage; 
    }
    
    public void lireListeRefParams(XMLTree noeud_affichage, Affichage affichage) {
        Vector listes = new Vector(); // listes de rfrences, Vector de tableau de String
        // toutes les listes de rfrences doivent avoir le mme nombre de rfrences
        int nbrefs = 0;
        for (XMLTree n = noeud_affichage.first_child; n != null; n = n.next_brother)
            if ("LISTEREFPARAMS".equals(n.tag)) {
                String[] liste = new String[n.getChildrenCount()]; // liste de rfrences
                for (int i=0; i<liste.length; i++) // boucle sur les lments REFPARAM
                    liste[i] = n.getChildAt(i).getAttVal("nom");
                listes.addElement(liste);
                if (nbrefs == 0)
                    nbrefs = liste.length;
                else if (nbrefs != liste.length)
                    System.err.println("Erreur: pas le mme nombre de rfrences dans toutes les listes de rfrences de paramtres");
            }
        if (listes.size() > 0) {
            affichage.params = new String[listes.size()][nbrefs];
            listes.copyInto(affichage.params);
        }
    }
    
    /** Sert  lire les attributs d'un paramtre d'affichage */
    public void parcourirParametreAffichage(XMLTree noeud_affichage, Affichage affichage) {
        String attribute_value = null;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("type") ) != null )
            affichage.type = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("titre") ) != null )
            affichage.titre = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("label") ) != null )
            affichage.label = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("xdebut") ) != null )
            affichage.xdebut = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("xfin") ) != null )
            affichage.xfin = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("ydebut") ) != null )
            affichage.ydebut = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("yfin") ) != null )
            affichage.yfin = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("xlog") ) != null )
            affichage.xlog = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("ylog") ) != null )
           affichage.ylog = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("fond") ) != null )
            affichage.fond = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("xdebutfond") ) != null )
            affichage.xdebutfond = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("xfinfond") ) != null )
            affichage.xfinfond = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("ydebutfond") ) != null )
            affichage.ydebutfond = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("yfinfond") ) != null )
            affichage.yfinfond = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("zooming") ) != null )
            affichage.zooming = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("imgdimx") ) != null )
            affichage.imgdimx = Integer.parseInt(attribute_value);
        
        if (  ( attribute_value = noeud_affichage.getAttVal("imgdimy") ) != null )
            affichage.imgdimy = Integer.parseInt(attribute_value);
        
        // palette ?
        
        if (  ( attribute_value = noeud_affichage.getAttVal("connecter") ) != null )
            affichage.connecter = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("ordreSousAff") ) != null )
// attention aux majuscules pour ordreSousAff (remplaces par le parser dans une ancienne version)
            affichage.ordreSousAff = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("boucle") ) != null )
            affichage.boucle = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("points") ) != null )
            affichage.points = attribute_value;
        
        if (  ( attribute_value = noeud_affichage.getAttVal("couleurs") ) != null )
            affichage.couleurs = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("histogramme") ) != null )
            affichage.histogramme = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("effacer") ) != null )
            affichage.effacer = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("ajuster") ) != null )
            affichage.ajuster = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("recadrer") ) != null )
            affichage.recadrer = Boolean.valueOf(attribute_value).booleanValue();
        
        if (  ( attribute_value = noeud_affichage.getAttVal("carre") ) != null )
            affichage.carre = Boolean.valueOf(attribute_value).booleanValue();  
    }
    
    /**
     * Traitement des actions de l'utilisateur
     */
    public void actionPerformed(ActionEvent e) {
        String cmd = e.getActionCommand();
        if (cmd.startsWith("OK")) {
            noaff = 0;
            try {
                noaff = (Integer.valueOf(cmd.substring(2))).intValue();
            } catch (NumberFormatException ex) {
            }
            startSimu(true);
        } else if ("Reset".equals(cmd)) {
            if (za != null && za.estLive())
                za.stop();
            initParams();
        } else if ("Relancer".equals(cmd)) {
            if (za != null && za.estLive())
                za.stop();
            startSimu(false);
        } else if (cmd.startsWith("help")) {
            int noparam = Integer.valueOf(cmd.substring(4)).intValue();
            aideParam(noparam);
        }
    }
    
    public void changerValeurs(ListeValeurs l) throws SimuException {
        Vector inp = params.getInParams();
        for (int i=0; i<inp.size(); i++) {
            ParamIn par = (ParamIn)inp.elementAt(i);
            String val;
            try {
                val = l.lireString(par.label);
            } catch (SimuException ex) {
                continue;
            }
            Object chp = champs.elementAt(i);
            if (chp instanceof TextField)
                ((TextField)chp).setText(val);
            else if (chp instanceof Scrollbar)
                ((Scrollbar)chp).setValue((new Double(val)).intValue());
            else if (chp instanceof Choice)
                ((Choice)chp).select(val);
            else if (chp instanceof Checkbox)
                ((Checkbox)chp).setState("on".equals(val));
        }
    }
    
    public void itemStateChanged(ItemEvent e) {
        // itemStateChanged est appel par chxens
        String nom = chxens.getSelectedItem();
        try {
            ListeValeurs l = (ListeValeurs)ensembles.lire(nom);
            changerValeurs(l);
        } catch (SimuException ex) {
            handleException(ex);
            return;
        }
    }
    
    /**
     * Traitement d'une erreur
     */
    public void handleException(SimuException ex) {
        System.err.println("SimuException: " + ex.getMessage());
        
        removeAll();
        setLayout(new BorderLayout());
        add(BorderLayout.NORTH, new Label(Messages.get("Erreur") + ": " + ex.getMessage()));
        Button breset = new Button(Messages.get("Reinitialiser"));
        breset.setActionCommand("Reset");
        breset.addActionListener(this);
        add(BorderLayout.SOUTH, breset);
        validate();
    }
    
    /**
     * Lecture des champs des paramtres d'entre
     */
    public ListeValeurs lireChamps() throws SimuException {
        ListeValeurs in = new ListeValeurs();
        Vector inp = params.getInParams();
        for (int i=0; i<inp.size(); i++) {
            ParamIn par = (ParamIn)inp.elementAt(i);
            String text = null;
            if (champs.elementAt(i) instanceof TextField) {
                TextField tf = (TextField)champs.elementAt(i);
                text = tf.getText();
                // test si le type de text est par.type
                if ("nombre".equals(par.type)) {
                    try {
                        Double d = new Double(text);
                    } catch (NumberFormatException ex) {
                        throw new SimuException(text + " " + Messages.get("pas_un_nombre"));
                    }
                }
            } else if (champs.elementAt(i) instanceof Choice) {
                Choice choix = (Choice)champs.elementAt(i);
                text = choix.getSelectedItem();
            } else if (champs.elementAt(i) instanceof Scrollbar) {
                Scrollbar curseur = (Scrollbar)champs.elementAt(i);
                text = Integer.toString(curseur.getValue());
            } else if (champs.elementAt(i) instanceof Checkbox) {
                Checkbox chk = (Checkbox)champs.elementAt(i);
                if (chk.getState())
                    text = "on";
                else
                    text = "off";
            }
            in.ajouterString(par.label, text);
        }
        if (chxens != null)
            selens = chxens.getSelectedItem();
        return(in);
    }
    
    // panel non modifiable des valeurs des paramtres d'entre
    /*
    public Panel makeStaticParamPanel(ListeValeurs in) {
        Vector inp = params.getInParams();
        Panel pval1 = new Panel();
        pval1.setLayout(new BorderLayout());
        Panel pval = new Panel();
        pval.setLayout(new GridLayout(inp.size(), 1));
        pval1.add("North", pval);
        for (int i=0; i<inp.size(); i++) {
            ParamIn par = (ParamIn)inp.elementAt(i);
            String text = in.lireString(par.label);
            String text2 = par.titre + ": " + text;
            if (par.unit != null)
                text2 += " " + par.unit;
            pval.add(new Label(text2));
        }
        return(pval1);
    }
    */
    
    /**
     * vrification que les paramtres de l'affichage sont tous dclars comme ParamOut
     */
    protected void verifParametres(Affichage aff, Vector outp) {
        if (aff.params != null)
            for (int ip=0; ip<aff.params.length; ip++)
                for (int jp=0; jp<aff.params[ip].length; jp++) {
                    String nomp = aff.params[ip][jp];
                    boolean trouv = false;
                    for (int io=0; io<outp.size(); io++)
                        if (nomp != null && nomp.equals(((ParamOut)outp.elementAt(io)).label))
                            trouv = true;
                    if (!trouv)
                        System.err.println(Messages.get("parametre_affichage") + " " + nomp +
                            " " + Messages.get("pas_declare"));
                }
        if (aff.sousAff != null)
            for (int ia=0; ia<aff.sousAff.length; ia++)
                verifParametres(aff.sousAff[ia], outp);
    }
    
    /**
     * Lancement de la simulation
     */
    public void startSimu(boolean init) {
        ListeValeurs in;
        try {
            in = lireChamps();
        } catch (SimuException ex) {
            handleException(ex);
            return;
        }
        Vector outp = params.getOutParams();
        Vector vaff = params.getAffichage();
        Affichage aff = (Affichage)vaff.elementAt(noaff);
        
        if (init) {
            verifParametres(aff, outp);
            //Panel pval1 = makeStaticParamPanel(in);
            Panel pval1 = makeInputPanel(in);
            removeAll();
            setLayout(new BorderLayout());
            add(BorderLayout.WEST, pval1);
        }
        
        try {
            if (aff == null)
                throw new SimuException(Messages.get("affichage_non_defini"));
            if (init || za == null)
                za = creationTypeAffichage(aff, false);
            Chronos urgent = null;
            if (za.estLive()) {
                urgent = new Chronos(this, za, aff.boucle);
                za.setChronos(urgent);
                urgent.initCalculLive(in);
            }
            if (init)
                add(BorderLayout.CENTER, za.initPanel(outp));
            za.affiche(in, outp);
        } catch (SimuException ex) {
            handleException(ex);
            return;
        }
        
        if (init) {
            Button breset = new Button(Messages.get("Reinitialiser"));
            breset.setActionCommand("Reset");
            breset.addActionListener(this);
            add(BorderLayout.SOUTH, breset);
            if (scrollp != null) {
                scrollp.setSize(scrollp.getComponent(0).getPreferredSize().width + scrollp.getVScrollbarWidth(),
                    getSize().height - breset.getSize().height);
                validate();
                scrollp.invalidate();
            }
            validate();
        }
    }
    
    public TypeAffichage creationTypeAffichage(Affichage aff, boolean sousaff) throws SimuException {
        TypeAffichage ta;
        if ("plotlive".equals(aff.type))
            ta = new AffPlotLive(this, aff, sousaff);
        else if ("plot".equals(aff.type))
            ta = new AffPlot(this, aff, sousaff);
        else if ("tableau".equals(aff.type))
            ta = new AffPlotTab(this, aff, sousaff);
        else if ("image".equals(aff.type))
            ta = new AffImage(this, aff, sousaff, true);
        else if ("imagestatique".equals(aff.type))
            ta = new AffImage(this, aff, sousaff, false);
        else if ("multiplot".equals(aff.type))
            ta = new AffMulti(this, aff, sousaff);
        else
            throw new SimuException(Messages.get("affichage_inconnu") + ": " + aff.type);
        return(ta);
    }
    
    /**
     * titres des axes
     */
    public static String[] titresAxes(Affichage aff, Vector outp) {
        if (aff == null)
            return null;
        if (aff.params == null)
            return null;
        String sx = null, sy = null;
        String lsx = aff.params[0][0];
        String lsy = aff.params[0][1];
        sx = lsx;
        sy = lsy;
        for (int i=0; i<outp.size(); i++) {
            ParamOut par = (ParamOut)outp.elementAt(i);
            if (lsx.equals(par.label)) {
                if (par.titre != null)
                    sx = par.titre;
            }
            if (lsy.equals(par.label)) {
                if (par.titre != null)
                    sy = par.titre;
            }
        }
        String[] res = new String[2];
        res[0] = sx;
        res[1] = sy;
        return(res);
    }
    
    /**
     * Renvoit l'aide pour le paramtre indiqu  partir du fichier XML,
     * ou null s'il n'y a pas d'aide pour ce paramtre
     */
    protected String lireAideParam(ParamIn param) {
        if (param.description != null)
            return(param.description);
        
        if (xmlDoc == null)
            return(null);
        
        String aide = null;
        
        XMLTree appletdoc = null;
        if ("APPLET_SIMULAB".equals(xmlDoc.tag))
            appletdoc = xmlDoc;
        else
            appletdoc = xmlDoc.getNode("APPLETDOC");
        if (appletdoc == null)
            System.err.println(Messages.get("sans_appletdoc"));
        else {
            XMLTree listeparams = appletdoc.getNode("LISTEPARAMS");
            if (listeparams == null)
                System.err.println(Messages.get("sans_listeparams"));
            else {
                XMLTree node;
                for (node=listeparams.first_child; node!=null; node=node.next_brother) {
                    if (node.tag.equals("PARAMETRE")) {
                        if (node.getAttVal("label").equals(param.label))
                            aide = node.val;
                    } else if (node.tag.equals("PARAMETRE_ENTREE")) {
                        if (node.getAttVal("label").equals(param.label))
                            if (node.getNode("DESCRIPTION") != null)
                                aide = node.getNode("DESCRIPTION").val;
                    }
                }
            }
        }
        if (aide != null)
            param.description = aide;
        return(aide);
    }
    
    /**
     * Affiche l'aide pour le paramtre indiqu  partir du fichier XML.
     * On suppose ici que le fichier XML est correct (le test est fait ailleurs).
     */
    public void aideParam(int nop) {
        if (xmlDoc == null)
            lireXML();
        ParamIn par = (ParamIn)params.getInParams().elementAt(nop);
        String aide = lireAideParam(par);
        String label = par.label;
        if (aide == null)
            System.err.println(Messages.get("sans_parametre") + " '" + label + "' " + Messages.get("dans_xml"));
        if (aide != null) {
            InfoWindow f = new InfoWindow(Messages.get("parametre") + " " + label, aide);
            Point apppoint = getLocationOnScreen();
            f.setLocation(apppoint.x+50, apppoint.y+50);
            f.setVisible(true);
        }
    }
    
    /**
    * Vrifie l'existence de l'aide dans le ficher de configuration XML
    * @return true si l'aide existe, false sinon
    */
    protected boolean verifExistenceAide(ParamIn param) {
        String aide = lireAideParam(param);
        return(aide != null && !"".equals(aide));
    }
    
    /**
     * renvoit le double correspondant au String
     */
    public static double getDouble(String sd) {
        double d;
        d = (new Double(sd)).doubleValue();
        return(d);
    }

    /**
     * pour bloquer le programme pendant nms millisecondes
     */
    public void dormir(long nms) {
        try {
            Thread.currentThread().sleep(nms);
        } catch (InterruptedException e) {}
    }
    
// *************************************
// variables et mthodes pour SimuParams

    /** liste des paramtres d'entre (Vector of ParamIn) */
    protected Vector inparams = new Vector();
    
    /** liste des paramtres de sortie (Vector of ParamOut) */
    protected Vector outparams = new Vector();
    
    /** liste des possibilits d'affichage (Vector of Affichage) */
    protected Vector affichages = new Vector();
    
    /** liste des ensembles de valeurs par dfaut */
    protected ListeValeurs ensembles = new ListeValeurs();
    
    /** titre des ensembles de valeurs */
    protected String titreEnsembles = Messages.get("Ensembles");
    
    /** renvoit la liste des paramtres d'entre (Vector of ParamIn) */
    public Vector getInParams() {
        return inparams;
    }

    /** renvoit la liste des paramtres de sortie (Vector of ParamOut) */
    public Vector getOutParams() {
        return outparams;
    }

    /** renvoit la liste des possibilits d'affichage (Vector of Affichage) */
    public Vector getAffichage() {
        return affichages;
    }
    
    /** renvoit l'affichage avec le label donn */
    public Affichage getAffichage(String label) {
        if (label == null)
            return(null);
        for (int i=0; i<affichages.size(); i++)
            if (label.equals(((Affichage)affichages.elementAt(i)).label))
                return((Affichage)affichages.elementAt(i));
        for (int i=0; i<affichages.size(); i++) {
            Affichage affi = (Affichage)affichages.elementAt(i);
            if (affi.sousAff != null) {
                for (int j=0; j<affi.sousAff.length; j++)
                    if (label.equals(affi.sousAff[j].label))
                        return(affi.sousAff[j]);
            }
        }
        return(null);
    }
    
    /** renvoit l'affichage choisi par l'utilisateur */
    public Affichage getAffichageChoisi() {
        if (noaff == -1)
            return(null);
        return((Affichage)affichages.elementAt(noaff));
    }
    
    /** ajoute un paramtre d'entre  la liste */
    public void addParamIn(ParamIn p) {
        getInParams().addElement(p);
    }

    /** ajoute un paramtre de sortie  la liste */
    public void addParamOut(ParamOut p) {
        getOutParams().addElement(p);
    }

    /** ajoute une option d'affichage  la liste */
    public void addAffichage(Affichage p) {
        getAffichage().addElement(p);
    }

    /** ajoute un ensemble de valeurs */
    public void addEnsemble(String nom, ListeValeurs l) {
        ensembles.ajouter(nom, l);
    }
    
    /** Dfinit le titre des ensembles de valeurs */
    public void setTitreEnsembles(String titre) {
        titreEnsembles = titre;
    }
    
    /** Spcifie le fichier XML avec la dfinition des paramtres et des affichages. */
    public void appletDoc(String nomFichier) {
        nomFichierXML = nomFichier;
    }
    
    public void lireXML() {
        XMLTree tree = new XMLTree(null, "", null, "");
        URL hurle;
        URL docbase = null;
        try {
            docbase = getDocumentBase();
        } catch (Exception ex) { // il y a un bug dans getDocumentBase => on ignore les erreurs
            System.err.println("Erreur: getDocumentBase() dconne !");
            return;
        }
        try {
            hurle = new URL(docbase, nomFichierXML);
        } catch (MalformedURLException ex) {
            System.err.println("MalformedURLException: " + ex.getMessage());
            return;
        }
        InputStream is = null;
        try {
            is = hurle.openStream();
        } catch (IOException ex) {
            // si a marche pas, c'est peut-tre parce-qu'on a mis le fichier XML dans une
            // archive jar et qu'il faut aller le chercher l
            is = getClass().getClassLoader().getResourceAsStream(nomFichierXML);
            if (is == null) {
                System.err.println(Messages.get("ressource_introuvable") + ": " + nomFichierXML);
                return;
            }
        }
        Parser_XML p = new Parser_XML(is, System.out, tree);
        p.parse(0);
        if (!"DEADOC".equals(tree.tag) && !"APPLET_SIMULAB".equals(tree.tag)) {
            System.err.println(Messages.get("mauvais_xml"));
            return;
        }
        xmlDoc = tree;
    }
    
    public void initCalculLive(ListeValeurs in) throws SimuException {
    }

    public ListeValeurs calculLive() throws SimuException {
        throw new SimuException(Messages.get("calculLive_necessaire"));
    }

    public ListeValeurs calcul(ListeValeurs in) throws SimuException {
        throw new SimuException(Messages.get("calcul_necessaire"));
    }
    
    class InfoWindow extends Frame implements ActionListener {
        public InfoWindow(String title, String message) {
            super(title);
            setLayout(new BorderLayout());
            int nblignes = 5;
            if (message.length() > 200)
                nblignes = 10;
            TextArea text = new TextArea(message, nblignes, 60, TextArea.SCROLLBARS_VERTICAL_ONLY);
            text.setEditable(false);
            add(BorderLayout.CENTER, text);
            Button bok = new Button(Messages.get("OK"));
            bok.addActionListener(this);
            Panel panel = new Panel();
            panel.add(bok);
            add(BorderLayout.SOUTH, panel);
            pack();
            addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    setVisible(false);
                    dispose();
                }
            });
        }
        public void actionPerformed(ActionEvent event) {
            setVisible(false);
            dispose();
        }
    }
}
