/*
FTS: Spectromtrie par TF

Copyright (C) 2002 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

import java.io.*;
import java.net.*;
import java.util.Vector;

import simu.*;

/**
 * Algorithme de simulation pour SimuApplet.
 * Spectromtrie par TF
 * compilation: javac -target 1.1 -encoding ISO-8859-1 -d build/FTS -classpath SimuLab.jar FTS.java
 */
public class FTS extends SimuApplet {
    
    Affichage graphe1;
    Affichage graphe2;
    String dossier;
    
    public void init() {
        setTitreEnsembles("exemples");
        // lecture des paramtres de l'applet
        // syntaxe: "fichier1,fichier2,...,fichiern"
        dossier = getParameter("dossier");
        String params = getParameter("fichiers");
        int indv = params.indexOf(',');
        boolean dernier = false;
        Vector vFichiers = new Vector();
        while (!dernier) {
            String nomf;
            if (indv == -1) {
                dernier = true;
                nomf = params.trim();
            } else {
                nomf = params.substring(0,indv).trim();
                params = params.substring(indv+1);
                indv = params.indexOf(',');
            }
            String cheminf;
            if (dossier != null)
                cheminf = dossier + '/' + nomf;
            else
                cheminf = nomf;
            vFichiers.addElement(nomf);
            EnteteFichier ent = lireEnteteFichier(cheminf);
            ListeValeurs lv = new ListeValeurs();
            lv.ajouterString("fspectre", nomf);
            lv.ajouterDouble("s1", ent.bornemin);
            lv.ajouterDouble("s2", ent.bornemax);
            //System.out.println("addEnsemble " + (new File(nomf)).getName());
            addEnsemble((new File(nomf)).getName(), lv);
        }
        String[] tFichiers = new String[vFichiers.size()];
        vFichiers.copyInto(tFichiers);
        /*
        ListeValeurs lv = new ListeValeurs();
        lv.ajouterString("fspectre", "4000K7.txt");
        lv.ajouterDouble("s1", 20480);
        lv.ajouterDouble("s2", 20660);
        addEnsemble("4000K7", lv);
        lv = new ListeValeurs();
        lv.ajouterString("fspectre", "5000K2.txt");
        lv.ajouterDouble("s1", 20480);
        lv.ajouterDouble("s2", 20660);
        addEnsemble("5000K2", lv);
        lv = new ListeValeurs();
        lv.ajouterString("fspectre", "6000F9.txt");
        lv.ajouterDouble("s1", 20480);
        lv.ajouterDouble("s2", 20660);
        addEnsemble("6000F9", lv);
        lv = new ListeValeurs();
        lv.ajouterString("fspectre", "6000F9na.txt");
        lv.ajouterDouble("s1", 16860);
        lv.ajouterDouble("s2", 17140);
        addEnsemble("6000F9na", lv);
        lv = new ListeValeurs();
        lv.ajouterString("fspectre", "7000F2.txt");
        lv.ajouterDouble("s1", 20480);
        lv.ajouterDouble("s2", 20660);
        addEnsemble("7000F2v", lv);
        lv = new ListeValeurs();
        lv.ajouterString("fspectre", "8000A6.txt");
        lv.ajouterDouble("s1", 20480);
        lv.ajouterDouble("s2", 20660);
        addEnsemble("8000A6", lv);
        */
       
        ParamIn fspectre = new ParamIn();
        fspectre.type = "string";
        fspectre.label = "fspectre";
        fspectre.titre = "fichier du spectre";
        fspectre.acquisition = "choix";
        /*String[] tchoix = {"4000K7.txt", "5000K2.txt", "6000F9.txt",
                           "6000F9na.txt", "7000F2.txt", "8000A6.txt"};*/
        //fspectre.choix = tchoix;
        fspectre.choix = tFichiers;
        //fspectre.defaut = "6000F9na.txt";
        fspectre.defaut = tFichiers[0];
        addParamIn(fspectre);
        
        ParamIn s1 = new ParamIn();
        s1.type = "nombre";
        s1.label = "s1";
        s1.titre = "borne min du filtre";
        s1.unit = "cm-1";
        s1.acquisition = "champ";
        s1.defaut = "16930";
        addParamIn(s1);
        ParamIn s2 = new ParamIn();
        s2.type = "nombre";
        s2.label = "s2";
        s2.titre = "borne max du filtre";
        s2.unit = "cm-1";
        s2.acquisition = "champ";
        s2.defaut = "17010";
        addParamIn(s2);
        
        ParamIn D = new ParamIn();
        D.type = "nombre";
        D.label = "D";
        D.titre = "diff de marche maximale";
        D.unit = "cm";
        D.acquisition = "champ";
        D.defaut = "0.1";
        addParamIn(D);
        
        ParamIn calculerN = new ParamIn();
        calculerN.type = "string";
        calculerN.label = "calculerN";
        calculerN.titre = "calculer le nb de points";
        calculerN.acquisition = "case";
        calculerN.defaut = "off";
        addParamIn(calculerN);
        
        ParamIn N = new ParamIn();
        N.type = "nombre";
        N.label = "N";
        N.titre = "nb de pts ds l'interfrogramme";
        N.acquisition = "champ";
        N.defaut = "300";
        addParamIn(N);
        
        // paramtres de sortie
        ParamOut ddm = new ParamOut();
        ddm.type = "nombre";
        ddm.label = "ddm";
        ddm.titre = "ddm (cm)";
        addParamOut(ddm);
        
        ParamOut amplitude = new ParamOut();
        amplitude.type = "nombre";
        amplitude.label = "amplitude";
        amplitude.titre = "";
        addParamOut(amplitude);
        
        ParamOut s0_1 = new ParamOut();
        s0_1.type = "nombre";
        s0_1.label = "s0_1";
        s0_1.titre = "nombre d'onde (cm-1)";
        addParamOut(s0_1);
        
        ParamOut densite_1 = new ParamOut();
        densite_1.type = "nombre";
        densite_1.label = "densite_1";
        densite_1.titre = "";
        addParamOut(densite_1);
        
        ParamOut s0_2 = new ParamOut();
        s0_2.type = "nombre";
        s0_2.label = "s0_2";
        addParamOut(s0_2);
        
        ParamOut densite_2 = new ParamOut();
        densite_2.type = "nombre";
        densite_2.label = "densite_2";
        addParamOut(densite_2);
        
        ParamOut info = new ParamOut();
        info.type = "string";
        info.label = "info";
        
        addParamOut(info);
        // options d'affichage
        Affichage multi = new Affichage();
        multi.type = "multiplot";
        multi.titre = "calcul";
        multi.sousAff = new Affichage[2];
        
        graphe1 = new Affichage();
        graphe1.type = "plot";
        graphe1.titre = "interfrogramme";
        String[][] pparams1 = {{"ddm","amplitude"}};
        graphe1.params = pparams1;
        graphe1.points = "none";
        graphe1.connecter = true;
        graphe1.effacer = true;
        graphe1.recadrer = true;
        multi.sousAff[0] = graphe1;
        
        graphe2 = new Affichage();
        graphe2.type = "plot";
        graphe2.titre = "spectres initial et calcul";
        String[][] pparams2 = {{"s0_2","densite_2"}, {"s0_1","densite_1"}};
        graphe2.params = pparams2;
        graphe2.points = "none";
        graphe2.connecter = true;
        graphe2.effacer = true;
        graphe2.recadrer = true;
        multi.sousAff[1] = graphe2;
        
        addAffichage(multi);
        super.init();
}


// calcul en image (utilis avec affichage image)
    public void initCalculLive(ListeValeurs in) throws SimuException {
        throw new SimuException("calculLive n'est pas implment dans FTS: " +
            "utiliser calcul  la place");
    }
    
    public ListeValeurs calculLive() throws SimuException {
        throw new SimuException("calculLive n'est pas implment dans FTS: " +
            "utiliser calcul  la place");
    }
    
    public ListeValeurs calcul(ListeValeurs in) throws SimuException {
        ListeValeurs out = new ListeValeurs();
        String fspectre = in.lireString("fspectre");
        double s1 = in.lireDouble("s1");
        double s2 = in.lireDouble("s2");
        double dm = in.lireDouble("D");
        String calculerN = in.lireString("calculerN");
        int n;
        if ("on".equals(calculerN))
            n = trouverN(s1, s2, dm);
        else
            n = in.lireEntier("N");
        double sc = (n-1)/(2*dm);
        double borne1 = Math.floor(s1/sc)*sc;
        double borne2 = borne1 + sc;
        out.ajouterString("info", "nb de points: " + n +
            "  borne1: " + borne1 + "  borne2: " + borne2);
        
        // lecture du fichier
        String cheminf;
        if (dossier != null)
            cheminf = dossier + '/' + fspectre;
        else
            cheminf = fspectre;
        double[][] intensites = lireFichier(cheminf);
        if (intensites == null)
            return(null);
        double[] s = intensites[0];
        double[] fs = intensites[1];
        
        // cration d'un tableau des diffrences de marche
        // calcul  partir de D et N (N points, ddm entre 0 et D)
        double[] ddm = new double[n];
        for (int i=0; i<n; i++)
            ddm[i] = dm*(double)i/(n-1);
        
        // filtrage entre s1 et s2
        // -> rsultat = ss et ff
        Vector res = new Vector();
        res = simufiltre(s, fs, s1, s2);
        if (res == null)
            return(null);
        double[] ss = (double[])res.elementAt(0);
        double[] ff = (double[])res.elementAt(1);
        int ns = ss.length;
        
        // TF
        double[] fff = new double[ns];
        double[] intrf = new double[n];
        for (int i=0; i<ns; i++)
            fff[i] = ff[i];
        for (int i=0; i<n; i++)
            intrf[i] = interf_spec(ss, fff, ddm[i]);
        
        // TF
        double[] spectf = new double[ns];
        for (int i=0; i<ns; i++)
            spectf[i] = interf_spec(ss[i], intrf, ddm);
        
        // moyenne
        double min = spectf[0]/n;
        for (int i=0; i<ns; i++) {
            spectf[i] = spectf[i]/n;
            if (spectf[i] < min)
                min = spectf[i];
        }
        // solution peu elegante mais efficace pour la moyenne
        for (int i=0; i<ns; i++)
            spectf[i] -= min;
        double med_spectf = median(spectf);
        double med_ff = median(ff);
        double facteur = med_ff / med_spectf;
        for (int i=0; i<ns; i++)
            spectf[i] = spectf[i] * facteur;
        
        // sortie
        graphe1.xdebut = "0";
        graphe1.xfin = "" + dm;
        //graphe1.ydebut = "" + min(intrf);
        //graphe1.yfin = "" + max(intrf);
        //le dbut est coup
        int icoupe = intrf.length/6;
        min = intrf[icoupe];
        double max = intrf[icoupe];
        for (int i=icoupe+1; i<intrf.length; i++) {
            if (intrf[i] < min)
                min = intrf[i];
            if (intrf[i] > max)
                max = intrf[i];
        }
        graphe1.ydebut = "" + min;
        graphe1.yfin = "" + max;
        graphe2.xdebut = "" + min(ss);
        graphe2.xfin = "" + max(ss);
        graphe2.ydebut = "" + min(ff);
        graphe2.yfin = "" + max(spectf);
        
        for (int i=icoupe; i<n; i++)
            out.ajouterDouble("ddm", ddm[i]);
        for (int i=icoupe; i<n; i++)
            out.ajouterDouble("amplitude", intrf[i]);
        for (int i=0; i<ns; i++)
            out.ajouterDouble("s0_1", ss[i]);
        for (int i=0; i<ns; i++)
            out.ajouterDouble("densite_1", ff[i]);
        for (int i=0; i<ns; i++)
            out.ajouterDouble("s0_2", ss[i]);
        for (int i=0; i<ns; i++)
            out.ajouterDouble("densite_2", spectf[i]);
        return out;
    }
    
    /**
     * lire un fichier ASCII venant d'IDL,
     * avec le nombre d'lments sur la premire ligne
     * et " x  y" sur les suivantes -> t[0][i] = x; t[1][i] = y;
     */
    /*
    public double[][] lireFichier(String nomFichier) {
        try {
            URL url = new URL(getDocumentBase(), nomFichier);
            BufferedReader in = new BufferedReader(new InputStreamReader(url.openStream()));
            String ligne = in.readLine();
            int n = Integer.parseInt(ligne.trim());
            double[][] t = new double[2][n];
            for (int i=0; i<n && ligne != null; i++) {
                ligne = in.readLine();
                if (ligne != null) {
                    ligne = ligne.trim();
                    int is = ligne.indexOf(' ');
                    String sx = ligne.substring(0,is).trim();
                    String sy = ligne.substring(is+1).trim();
                    double x = (Double.valueOf(sx)).doubleValue();
                    double y = (Double.valueOf(sy)).doubleValue();
                    t[0][i] = x;
                    t[1][i] = y;
                }
            }
            return(t);
        } catch (MalformedURLException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (NumberFormatException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (IOException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        }
    }
    */
    
    /**
     * lire un fichier ASCII venant d'IDL,
     * avec le nombre d'lments sur la premire ligne
     * les bornes sur la deuxime ligne
     * la position gnrale du spectre (ignore) sur la troisime ligne
     * et " x  y" normaliss sur les suivantes
     * -> t[0][i] = x + bornemin;
     *    t[1][i] = y;
     */
    /*
    public double[][] lireFichier(String nomFichier) {
        try {
            URL url = new URL(getDocumentBase(), nomFichier);
            BufferedReader in = new BufferedReader(new InputStreamReader(url.openStream()));
            String ligne = in.readLine();
            int n = Integer.parseInt(ligne.trim());
            ligne = in.readLine();
            ligne = ligne.trim();
            int is = ligne.indexOf(' ');
            String sx = ligne.substring(0,is).trim();
            String sy = ligne.substring(is+1).trim();
            double bornemin = (Double.valueOf(sx)).doubleValue();
            double bornemax = (Double.valueOf(sy)).doubleValue();
            ligne = in.readLine(); // la position
            double[][] t = new double[2][n];
            double x,y;
            for (int i=0; i<n && ligne != null; i++) {
                ligne = in.readLine();
                if (ligne != null && !"".equals(ligne)) {
                    ligne = ligne.trim();
                    is = ligne.indexOf(' ');
                    sx = ligne.substring(0,is).trim();
                    sy = ligne.substring(is+1).trim();
                    x = (Double.valueOf(sx)).doubleValue() + bornemin;
                    y = (Double.valueOf(sy)).doubleValue();
                    t[0][i] = x;
                    t[1][i] = y;
                }
            }
            return(t);
        } catch (MalformedURLException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (NumberFormatException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (IOException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        }
    }
    */
    
    class EnteteFichier {
        public String titre;
        public double bornemin, bornemax;
    }
    
    /**
     * lire l'entte d'un fichier ASCII venant d'IDL,
     * avec le nombre d'lments sur la premire ligne
     * les bornes  et l'incrment inc sur la deuxime ligne
     * la position gnrale du spectre (ignore) sur la troisime ligne
     * et "y" normalis sur les suivantes
     */
    public EnteteFichier lireEnteteFichier(String nomFichier) {
        try {
            URL url = new URL(getDocumentBase(), nomFichier);
            BufferedReader in = new BufferedReader(new InputStreamReader(url.openStream()));
            String ligne = in.readLine();
            ligne = in.readLine();
            ligne = ligne.trim();
            int is = ligne.indexOf(' ');
            String sx = ligne.substring(0,is).trim();
            double bornemin = (Double.valueOf(sx)).doubleValue();
            ligne = ligne.substring(is+1).trim();
            is = ligne.indexOf(' ');
            String sy = ligne.substring(0,is).trim();
            double bornemax = (Double.valueOf(sy)).doubleValue();
            ligne = ligne.substring(is+1).trim();
            double increment = (Double.valueOf(ligne)).doubleValue();
            ligne = in.readLine();
            String titrepos = ligne.trim();
            in.close();
            EnteteFichier ent = new EnteteFichier();
            ent.titre = titrepos;
            ent.bornemin = bornemin;
            ent.bornemax = bornemax;
            return(ent);
        } catch (MalformedURLException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (NumberFormatException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (IOException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        }
    }
    
    /**
     * lire les donnes d'un fichier ASCII venant d'IDL,
     * avec le nombre d'lments sur la premire ligne
     * les bornes  et l'incrment inc sur la deuxime ligne
     * la position gnrale du spectre (ignore) sur la troisime ligne
     * et "y" normalis sur les suivantes
     * -> t[0][i] = bornemin + increment*i;
     *    t[1][i] = y;
     *    t[0][t[0].length-1] = bornemax; (bornemin + inc*n risque d'tre trop grand)
     */
    public double[][] lireFichier(String nomFichier) {
        try {
            URL url = new URL(getDocumentBase(), nomFichier);
            BufferedReader in = new BufferedReader(new InputStreamReader(url.openStream()));
            String ligne = in.readLine();
            int n = Integer.parseInt(ligne.trim());
            ligne = in.readLine();
            ligne = ligne.trim();
            int is = ligne.indexOf(' ');
            String sx = ligne.substring(0,is).trim();
            double bornemin = (Double.valueOf(sx)).doubleValue();
            ligne = ligne.substring(is+1).trim();
            is = ligne.indexOf(' ');
            String sy = ligne.substring(0,is).trim();
            double bornemax = (Double.valueOf(sy)).doubleValue();
            ligne = ligne.substring(is+1).trim();
            double increment = (Double.valueOf(ligne)).doubleValue();
            ligne = in.readLine();
            double[][] t = new double[2][n];
            double y;
            for (int i=0; i<n && ligne != null; i++) {
                ligne = in.readLine();
                if (ligne != null && !"".equals(ligne)) {
                    ligne = ligne.trim();
                    y = (Double.valueOf(ligne)).doubleValue();
                    t[0][i] = bornemin + increment*i;
                    t[1][i] = y;
                }
            }
            in.close();
            // pour viter les erreurs d'arrondi
            t[0][t[0].length-1] = bornemax;
            return(t);
        } catch (MalformedURLException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (NumberFormatException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        } catch (IOException ex) {
            System.err.println(ex.getClass().getName() + " : " + ex.getMessage());
            return(null);
        }
    }
    
    /**
     * simule un filtre, de bande passante [s1, s2]
     */
    public static Vector simufiltre(double[] s, double[] f, double s1, double s2) {
        int[] ind = where(s, s1, s2);
        if (ind == null) {
            System.err.println("simufiltre: Ca coince (s1="+s1+" s2="+s2+")");
            return(null);
        }
        int n = ind[1] - ind[0] + 1;
        double[] ss = new double[n];
        double[] ff = new double[n];
        for (int i=0; i<n; i++) {
            ss[i] = s[i+ind[0]];
            ff[i] = f[i+ind[0]];
        }
        
        // epaisseur des fronts montant et descendant
        double ds = (s2 - s1) / 4.5;
        
        // filtre = 1 partout, sauf aux bords
        double[] fil = new double[n];
        for (int i=0; i<n; i++)
            fil[i] = 1;
        int i1 = 0;
        int i2 = n-1;
        for (int i=0; i<n; i++) {
            if (ss[i] < s1 + ds)
                i1 = i;
            if (i2 == n-1 && ss[i] > s2 - ds)
                i2 = i;
        }
        for (int i=0; i<=i1; i++)
            fil[i] = (1 + Math.cos(Math.PI * (ss[i] - s1)/ds - Math.PI) ) / 2;
        for (int i=i2; i<n; i++)
            fil[i] = (1 + Math.cos(Math.PI * (s2 - ss[i])/ds - Math.PI) ) / 2;
        
        for (int i=0; i<n; i++)
            ff[i] = ff[i] * fil[i];
        
        Vector res = new Vector();
        res.addElement(ss);
        res.addElement(ff);
        return(res);
    }
    
    public static double interf_spec(double[] s, double[] f, double d) {
        int n = s.length;
        double[] phi = new double[n];
        for (int i=0; i<n; i++)
            phi[i] = 2 * Math.PI * s[i] * d;
        double som = 0;
        for (int i=0; i<n; i++)
            som += f[i] * Math.cos(phi[i]);
        return(som);
    }
    
    public static double interf_spec(double s, double[] f, double[] d) {
        int n = f.length;
        double[] phi = new double[n];
        for (int i=0; i<n; i++)
            phi[i] = 2 * Math.PI * s * d[i];
        double som = 0;
        for (int i=0; i<n; i++)
            som += f[i] * Math.cos(phi[i]);
        return(som);
    }
    
    public static double median(double[] t) {
        double[] tc = new double[t.length];
        for (int i=1; i<t.length; i++)
            tc[i] = t[i];
        quicksort(tc);
        return(tc[t.length/2]);
    }
    
    public static double min(double[] t) {
        double min = t[0];
        for (int i=1; i<t.length; i++)
            if (t[i] < min)
                min = t[i];
        return(min);
    }
    
    public static double max(double[] t) {
        double max = t[0];
        for (int i=1; i<t.length; i++)
            if (t[i] > max)
                max = t[i];
        return(max);
    }
    
    public static void quicksort(double[] t){
        quicksort(t, 0, t.length-1);
    }

    private static void quicksort(double[] t, int lower, int upper){
        int i, m;
        double temp, pivot;
        
        if (lower < upper) {
            temp =  t[lower];
            t[lower] = t[(upper+lower)/2];
            t[(upper+lower)/2] = temp;
            pivot = t[lower];
            m = lower;
            for (i = lower + 1; i <= upper; i++)
                if (t[i] < pivot) {
                    m++;
                    temp = t[m];
                    t[m] = t[i];
                    t[i] = temp;
                }
            temp = t[m];
            t[m] = t[lower];
            t[lower] = temp;
            quicksort (t, lower, m - 1);
            quicksort (t, m + 1, upper);
        }
    }
    
    public int trouverN(double s1, double s2, double dm) {
        /*
        // CHOIX DU PARAMETRE OPTIMAL
        // = recherche d'un produit d'entier
        double test = 10;
        
        // choix complexe de la limite, dependant un peu de dm et s1
        double limtest = .00001 * (1./dm) * Math.pow(2.e4 / s1, 4.);
        
        // initialisation
        double ds = s2 - s1;  // largeur du spectre
        int n1 = (int)Math.floor(s1 / ds);
        int n10 = n1; // borne "a gauche"
        Vector nopv = new Vector(); // de Double
        nopv.addElement(new Double(0.5));
        double nop=-1;
        double sc=0;
        // -1- boucle pour la recherche d'un "bon entier"
        while (test > limtest && n1 >= n10/3) {
            sc = s1 / n1;
            nop = 2. * dm * sc;
            // test pour voir si nop est suffisament entier"
            test = Math.pow((nop - Math.floor(nop)), 2.);
            // on cherche plus a gauche au cas ou ca n'a pas marche
            n1 = n1 - 1;
            // on construit le vecteur des valeurs, pour la solution de rattrapage
            nopv.addElement(new Double(nop));
        }
        // -2- rattrapage, si on n'est pas sur d'avoir abouti au bon endroit
        // car on est parvenu au bout de la boucle prcedente sans aboutir
        // = recherche de la moins mauvaise valeur parmi celles testees
        if (test > limtest) {
            double[] testv = new double[nopv.size()];
            for (int i=0; i<nopv.size(); i++) {
                double nopvv = ((Double)nopv.elementAt(i)).doubleValue();
                testv[i] = Math.pow((nopvv - Math.floor(nopvv)), 2.);
            }
            int ind = where(testv, min(testv));
            // la bonne valeur est :
            nop = ((Double)nopv.elementAt(ind)).doubleValue();
            n1  = n10 - ind;
            sc = s1 / (n1+1);
            // print, 'Rattrapage'
        }
        
        n1 = n1 + 1; // on rattrape la derniere etape
        System.out.println("ds, sc : " + ds + " " + sc);
        System.out.println("Bornes : " + n1*sc + " " + s1 + " " + s2 + " " + (n1+1)*sc);
        System.out.println("Descente : " + n10 + " " + n1);
        
        System.out.println("*** Nbre de points optimal N : "+ (nop+1) +" ***");
        return((int)Math.round(nop+1));
        */
        
        // CHOIX DU PARAMETRE OPTIMAL
        // algorithme : calcul de la plus petite valeur de N possible, 
        // et iteration jusqu'a trouver le bon intervalle spectral libre
        // entourant les valeurs
        double ds = s2 - s1;  // largeur du spectre
        int nop0 = (int)Math.floor(2*dm*ds); // plus petit entier possible-1
        
        double b1 = 0.;
        double b2 = 0.;
        int nop = nop0;
        double sc = 0.;
        int n1;
        while (b2 < s2) {
            // on augmente N d'une unite
            nop = nop + 1;
            sc  = nop / (2.*dm);
            // on calcule les bornes inf et sup de l'intervalle
            n1 = (int)Math.floor(s1 / sc);
            b1 = n1 * sc;
            b2 = b1 + sc;
            // si b2 est assez grand, on conclut   
        }
        
        //System.out.println("ds, sc : " + ds + " " + sc);
        //System.out.println("Bornes : " + b1 + " " + s1 + " " + s2 + " " + b2);
        //System.out.println("Iteration : " + nop0 + " " + nop + " soit +" + (nop-nop0));
        
        nop = nop + 1; //!!!!!!!!!!!!!!!!!!!!!!!!!!
        //System.out.println("*** Nbre de points optimal N : " + nop + " ***");
        return(nop);
    }
    
    /**
     * intervalle o t est entre min et max (inclus)
     * on suppose t tri par ordre croissant
     */
    static public int[] where(double[] t, double min, double max) {
        int imin = -1;
        int imax = -1;
        for (int i=0; i<t.length; i++) {
            if (t[i] <= min)
                imin = i;
            if (t[i] >= max) {
                imax = i;
                break;
            }
        }
        if (imin == -1 || imax == -1)
            return(null);
        int[] res = {imin, imax};
        return(res);
    }
    
    /**
     * indice o t a la valeur indique
     * renvoit -1 si non trouv
     */
    static public int where(double[] t, double v) {
        for (int i=0; i<t.length; i++)
            if (t[i] == v)
                return(i);
        return(-1);
    }
    
}
