/*
Copyright (C) 2009 Observatoire de Paris

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package serveurmica;

public class Formule {
    private Parametre param;
    private EtudeDeCas edc;
    private SuiviEDC suivi;
    
    public Formule(final String formule, final EtudeDeCas edc, final SuiviEDC suivi) {
        this.edc = edc;
        this.suivi = suivi;
        param = parser(ajParentheses(formule));
        // on pourrait optimiser a en analysant les formules d'abord (avec parser), et en rutilisant les objets pour une EDC donne
        // evaluer prendrait alors le suivi en paramtre
    }
    
    public double evaluer() {
        Double d = param.evaluer();
        if (d == null)
            return(0);
        else
            return(d.doubleValue());
    }
    
    private double getValeurVariable(final String nom) {
        return(edc.getValeurVariable(nom, suivi));
    }
    
    private String ajParentheses(String s) {
        String sops = "^/*-+=<>";
        for (int iops=0; iops<sops.length(); iops++) {
            char cops = sops.charAt(iops);
            int indop = s.indexOf(cops);
            if (indop > 0 && "^*/-+=<>Ee".indexOf(s.charAt(indop-1)) != -1)
                indop = -1; // un - ou + aprs un E ou un oprateur n'est pas un oprateur !
            int nindop = indop;
            int im,ip;
            char cm=' ',cp=' ';
            int pp;
            boolean ajp;
            while (nindop != -1) {
                ajp = false;
                im = indop - 1;
                if (im >= 0)
                    cm = s.charAt(im);
                pp = 0;
                while (im >= 0 && (pp != 0 || cm != '(') &&
                        (pp != 0 || (sops.indexOf(cm) == -1 ||
                        (im > 0 && "^*/-+=<>Ee".indexOf(s.charAt(im-1)) != -1)))) {
                    if (cm == ')')
                        pp++;
                    else if (cm == '(')
                        pp--;
                    im--;
                    if (im >= 0)
                        cm = s.charAt(im);
                }
                if (im < 0 || sops.indexOf(cm) != -1)
                    ajp = true;
                ip = indop + 1;
                if (ip >= 0)
                    cp = s.charAt(ip);
                pp = 0;
                while (ip < s.length() && (pp != 0 || cp != ')') &&
                        (pp != 0 || (sops.indexOf(cp) == -1 ||
                        (ip > 0 && "^*/-+=<>Ee".indexOf(s.charAt(ip-1)) != -1)))) {
                    if (cp == '(')
                        pp++;
                    else if (cp == ')')
                        pp--;
                    ip++;
                    if (ip < s.length())
                        cp = s.charAt(ip);
                }
                if (ip >= s.length() || sops.indexOf(cp) != -1)
                    ajp = true;
                if (ajp) {
                    s = s.substring(0, im+1) + "(" + s.substring(im+1, ip) + ")" +
                        s.substring(ip);
                    indop++;
                }
                nindop = s.substring(indop+1).indexOf(cops);
                indop = nindop + indop+1;
            }
        }
        return(s);
    }
    
    private Parametre parser(String s) {
        String sops = "^/*-+=<>";
        if (s.charAt(0) == '(' && s.charAt(s.length()-1) == ')') {
            boolean retparok = true;
            // on vrifie si on peut retirer les parenthses qui entourent l'expression
            int pp = 0;
            for (int i=1; i<s.length()-1; i++) {
                if (s.charAt(i) == '(')
                    pp++;
                else if (s.charAt(i) == ')')
                    pp--;
                if (pp < 0) {
                    retparok = false;
                    break;
                }
            }
            if (retparok)
                s = s.substring(1, s.length()-1);
        }
        int indop = -1;
        int pp = 0;
        for (int i=0; i<s.length(); i++) {
            if (pp == 0 && sops.indexOf(s.charAt(i)) != -1 &&
                    (i < 2 || "Ee".indexOf(s.charAt(i-1)) == -1 || "0123456789".indexOf(s.charAt(i-2)) == -1)) {
                indop = i;
                break;
            } else if (s.charAt(i) == '(')
                pp++;
            else if (s.charAt(i) == ')')
                pp--;
        }
        if (indop == -1) {
            boolean nb = true;
            for (int i=0; i<s.length(); i++)
                if ("0123456789.,-+ ".indexOf(s.charAt(i)) == -1  &&
                        ((s.charAt(i) != 'E' && s.charAt(i) != 'e') ||
                        (i == 0 || "0123456789".indexOf(s.charAt(i-1)) == -1))) {
                    nb = false;
                    break;
                }
            if (nb) {
                double d;
                try {
                    d = (new Double(s)).doubleValue();
                } catch (NumberFormatException ex) {
                    d = 0;
                }
                return(new Nombre(d));
            } else {
                int indf = s.indexOf('(');
                if (indf != -1 && s.charAt(s.length()-1) == ')') {
                    Fonction fct = new Fonction(s.substring(0,indf));
                    int indv = 0;
                    Parametre p1 = null;
                    Parametre p2 = null;
                    s = s.substring(indf+1, s.length()-1);
                    indv = s.indexOf(',');
                    if (indv != -1) {
                        p1 = parser(s.substring(0,indv).trim());
                        s = s.substring(indv+1);
                        p2 = parser(s.trim());
                    } else
                        p1 = parser(s.trim());
                    return(new Expression(fct, p1, p2));
                } else
                    return(new Variable(s));
            }
        } else {
            Fonction op = new Fonction(s.charAt(indop));
            String s1 = s.substring(0,indop).trim();
            if ("".equals(s1))
                s1 = "0"; // -(2+3) = 0 - (2+3)
            Parametre p1 = parser(s1);
            String s2 = s.substring(indop+1).trim();
            Parametre p2 = parser(s2);
            return(new Expression(op, p1, p2));
        }
    }
    
    class Expression implements Parametre {
        Fonction fct;
        Parametre p1,p2;
        
        public Expression(Fonction fct, Parametre p1, Parametre p2) {
            this.fct = fct;
            this.p1 = p1;
            this.p2 = p2;
        }
        public Double evaluer() {
            if (p2 == null) {
                Double d1 = p1.evaluer();
                if (d1 == null)
                    return(null);
                return(fct.evaluer(d1.doubleValue(), 0));
            } else {
                Double d1 = p1.evaluer();
                Double d2 = p2.evaluer();
                if (d1 == null || d2 == null)
                    return(null);
                return(fct.evaluer(d1.doubleValue(), d2.doubleValue()));
            }
        }
    }
    
    interface Parametre {
        public abstract Double evaluer();
    }
    
    class Variable implements Parametre {
        String nom;
        
        public Variable(String nom) {
            this.nom = nom;
        }
        public Double evaluer() {
            if (nom == null)
                return(null);
            if ("e".equals(nom))
                return(new Double(Math.E));
            else if ("pi".equals(nom))
                return(new Double(Math.PI));
            return(new Double(getValeurVariable(nom)));
        }
    }
    
    static class Nombre implements Parametre {
        double valeur;
        
        public Nombre(double valeur) {
            this.valeur = valeur;
        }
        public Double evaluer() {
            return(new Double(valeur));
        }
    }
    
    class Fonction {
        int type;
        static final int addition = 1;
        static final int soustraction = 2;
        static final int multiplication = 3;
        static final int division = 4;
        static final int puissance = 5;
        static final int logdecimal = 6;
        static final int ln = 7;
        static final int exp = 8;
        static final int sin = 9;
        static final int cos = 10;
        static final int tan = 11;
        static final int asin = 12;
        static final int acos = 13;
        static final int atan = 14;
        static final int abs = 15;
        static final int sqrt = 16;
        static final int partie_entiere = 17;
        static final int partie_frac = 18;
        static final int egalite = 19;
        static final int inferieur = 20;
        static final int superieur = 21;
        
        // constructeur pour un oprateur
        public Fonction(char c) {
            switch (c) {
                case '+': type = addition; break;
                case '-': type = soustraction; break;
                case '*': type = multiplication; break;
                case '/': type = division; break;
                case '^': type = puissance; break;
                case '=': type = egalite; break;
                case '<': type = inferieur; break;
                case '>': type = superieur; break;
                default: type = 0; break;
            }
        }
        
        // constructeur pour une fonction
        public Fonction(String nom) {
            if ("log".equals(nom))
                type = logdecimal;
            else if ("ln".equals(nom))
                type = ln;
            else if ("exp".equals(nom))
                type = exp;
            else if ("sin".equals(nom))
                type = sin;
            else if ("cos".equals(nom))
                type = cos;
            else if ("tan".equals(nom))
                type = tan;
            else if ("asin".equals(nom))
                type = asin;
            else if ("acos".equals(nom))
                type = acos;
            else if ("atan".equals(nom))
                type = atan;
            else if ("abs".equals(nom))
                type = abs;
            else if ("sqrt".equals(nom))
                type = sqrt;
            else if ("pent".equals(nom) || "floor".equals(nom))
                type = partie_entiere;
            else if ("pfrac".equals(nom) || "fpart".equals(nom))
                type = partie_frac;
            else
                type = 0;
        }
        
        public Double evaluer(double v1, double v2) {
            double d;
            switch (type) {
                case addition: d = v1 + v2; break;
                case soustraction:  d = v1 - v2; break;
                case multiplication:  d = v1 * v2; break;
                case division:  d = v1 / v2; break;
                case puissance:  d = Math.pow(v1, v2); break;
                case logdecimal:  d = Math.log(v1)/Math.log(10); break;
                case ln:  d = Math.log(v1); break;
                case exp:  d = Math.exp(v1); break;
                case sin:  d = Math.sin(v1); break;
                case cos:  d = Math.cos(v1); break;
                case tan:  d = Math.tan(v1); break;
                case asin:  d = Math.asin(v1); break;
                case acos:  d = Math.acos(v1); break;
                case atan:  d = Math.atan(v1); break;
                case abs:  d = Math.abs(v1); break;
                case sqrt:  d = Math.sqrt(v1); break;
                case partie_entiere:  d = Math.floor(v1); break;
                case partie_frac:  d = v1 - Math.floor(v1); break;
                case egalite: d = (v1 == v2 ? 1 : 0); break;
                case inferieur: d = (v1 < v2 ? 1 : 0); break;
                case superieur: d = (v1 > v2 ? 1 : 0); break;
                default: return(null);
            }
            return(new Double(d));
        }
    }
    
}
