/*
Copyright (C) 2009 Observatoire de Paris

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package serveurmica;

import java.io.*;
import java.util.*;

import javax.servlet.http.HttpServletResponse;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Etude de cas
 */
public class EtudeDeCas {
    private File fichier_xml;
    private File fichier_xsl;
    private String label;
    private String titre;
    private File dossier_edc; // dossier de l'tude de cas, pas du fichier XML qui se trouve dans un sous-dossier
    private String description;
    private String nom_variable_progression;
    private HashMap<String, String> formules_variables;
    private ArrayList<File> contributions_xml; // fichiers XML des contributions
    private Hashtable<String, File> contribs_etapes; // label tape -> fichier contrib correspondant
    private Hashtable<String, File> contribs_sections; // label section -> fichier contrib correspondant
    private Hashtable<String, String> titres_etapes; // label tape -> titre
    private Hashtable<String, String> etape_section; // label section -> label tape correspondante
    private Hashtable<String, ArrayList<String>> labels_sections; // label tape -> liste des labels des sections de l'tape, dans l'ordre
    private ArrayList<String> etapes_bloquantes;
    private Hashtable<String, String> glossaire; // mot -> dfinition
    private Plan plan;
    private Bibliographie bibliographie;
    private Document bibliotheque;
    
    public EtudeDeCas(final File fichier_xml, final File fichier_xsl) throws ExceptionMICA {
        this.fichier_xml = fichier_xml;
        this.fichier_xsl = fichier_xsl;
        label = fichier_xml.getName();
        if (label.indexOf('.') != -1)
            label = label.substring(0, label.lastIndexOf('.'));
        dossier_edc = fichier_xml.getParentFile().getParentFile();
        
        final Document doc = OutilsXML.lectureDocumentXML(fichier_xml);
        final Element racine = doc.getDocumentElement();
        if (!"MICA".equals(racine.getLocalName()))
            throw new ExceptionMICA("Mauvais lment racine pour " + label, true);
        
        final Element edc = OutilsXML.premierEnfant(racine, "ETUDE_DE_CAS");
        if (edc == null)
            throw new ExceptionMICA("Aucun lment ETUDE_DE_CAS pour " + label);
        titre = edc.getAttribute("titre");
        final Element metadonnees = OutilsXML.premierEnfant(edc, "METADONNEES");
        if (metadonnees == null)
            throw new ExceptionMICA("Aucun lment METADONNEES pour " + label, true);
        final Element el_desc = OutilsXML.premierEnfant(metadonnees, "DESCRIPTION");
        if (el_desc == null)
            description = null;
        else
            description = OutilsXML.valeurElement(el_desc);
        
        final Element variables = OutilsXML.premierEnfant(edc, "VARIABLES");
        if (variables != null) {
            final Element ref_progression = OutilsXML.premierEnfant(variables, "REF_PROGRESSION");
            if (ref_progression != null)
                nom_variable_progression = ref_progression.getAttribute("nom");
            else
                nom_variable_progression = null;
            final ArrayList<Element> elements_variables = OutilsXML.enfants(variables, "VARIABLE");
            formules_variables = new HashMap<String, String>();
            for (Element v : elements_variables)
                formules_variables.put(v.getAttribute("nom"), v.getAttribute("formule"));
        } else {
            nom_variable_progression = null;
            formules_variables = null;
        }
        
        final Element el_plan = OutilsXML.premierEnfant(edc, "PLAN");
        if (el_plan != null)
            plan = new Plan(el_plan);
        else
            plan = null;
        contributions_xml = new ArrayList<File>();
        final Element contributions = OutilsXML.premierEnfant(edc, "CONTRIBUTIONS");
        if (contributions == null)
            throw new ExceptionMICA("Aucun lment CONTRIBUTIONS pour " + label, true);
        final ArrayList<Element> elements_contributions = OutilsXML.enfants(contributions, "REF_CONTRIBUTION");
        for (Element c : elements_contributions) {
            final String label_contribution = c.getAttribute("label");
            final File dossier_contrib = new File(dossier_edc, label_contribution);
            final File fichier_contrib = new File(dossier_contrib, label_contribution + ".xml");
            if (!fichier_contrib.exists())
                throw new ExceptionMICA("EtudeDeCas " + label + ": la contribution " + label_contribution + " n'existe pas", true);
            contributions_xml.add(fichier_contrib);
        }
        
        contribs_etapes = new Hashtable<String, File>();
        contribs_sections = new Hashtable<String, File>();
        titres_etapes = new Hashtable<String, String>();
        etape_section = new Hashtable<String, String>();
        labels_sections = new Hashtable<String, ArrayList<String>>();
        etapes_bloquantes = new ArrayList<String>();
        glossaire = new Hashtable<String, String>();
        bibliographie = new Bibliographie();
        bibliotheque = null;
        for (File fichier_contrib : contributions_xml) {
            final Document doc_contrib = OutilsXML.lectureDocumentXML(fichier_contrib);
            final Element racine_contrib = doc_contrib.getDocumentElement();
            if (!"MICA".equals(racine_contrib.getLocalName()))
                continue;
            final ArrayList<Element> etapes = OutilsXML.enfants(racine_contrib, "ETAPE");
            for (Element etape : etapes) {
                final String label_etape = etape.getAttribute("label");
                if (plan != null)
                    plan.setTitreEtape(label_etape, etape.getAttribute("titre"));
                titres_etapes.put(label_etape, etape.getAttribute("titre"));
                contribs_etapes.put(label_etape, fichier_contrib);
                if ("oui".equals(etape.getAttribute("bloquante")))
                    etapes_bloquantes.add(label_etape);
                final ArrayList<Element> sections = OutilsXML.enfants(etape, "SECTION");
                for (Element section : sections)
                    ajouterSection(section, fichier_contrib, label_etape);
                
                ajouterBibliotheque(etape, fichier_contrib);
                ajouterBibliographie(etape);
                ajouterGlossaire(etape);
            }
        }
        
    } // fin constructeur EtudeDeCas
    
    private void ajouterSection(final Element section, final File fichier_contrib, final String label_etape) {
        final String label_section = section.getAttribute("label");
        contribs_sections.put(label_section, fichier_contrib);
        etape_section.put(label_section, label_etape);
        ArrayList<String> sections_etape = labels_sections.get(label_etape);
        if (sections_etape == null) {
            sections_etape = new ArrayList<String>();
            labels_sections.put(label_etape, sections_etape);
        }
        sections_etape.add(label_section);
        final ArrayList<Element> sous_sections = OutilsXML.enfants(section, "SECTION");
        for (Element sous_section : sous_sections)
            ajouterSection(sous_section, fichier_contrib, label_etape);
        ajouterBibliographie(section);
        ajouterGlossaire(section);
    }
    
    private void ajouterBibliotheque(final Element parent, final File fichier_contrib) throws ExceptionMICA {
        final Element bibliotheque_locale = OutilsXML.premierEnfant(parent, "BIBLIOTHEQUE");
        if (bibliotheque_locale == null)
            return;
        if (bibliotheque == null) {
            bibliotheque = OutilsXML.nouveauDocumentDOM();
            final Element racine = bibliotheque.createElementNS(null, "BIBLIOTHEQUE");
            racine.setAttributeNS(null, "titre_edc", titre);
            bibliotheque.appendChild(racine);
        }
        final Element racine_bibliotheque = bibliotheque.getDocumentElement();
        final ArrayList<Element> elts_bibliotheque_locale = OutilsXML.enfants(bibliotheque_locale, "ELEMENT_BIBLIOTHEQUE");
        for (Element elt : elts_bibliotheque_locale) {
            final Element elt_importe = (Element)bibliotheque.importNode(elt, true);
            final Element zone_image = OutilsXML.premierEnfant(elt_importe, "ZONE_IMAGE");
            if (zone_image != null) {
                if (!"texte".equals(zone_image.getAttribute("localisation")))
                    zone_image.setAttributeNS(null, "localisation", "texte");
            }
            ajouterDimensionsImages(elt_importe, fichier_contrib);
            racine_bibliotheque.appendChild(elt_importe);
        }
    }
    
    private void ajouterBibliographie(final Element parent) {
        final Element el_bibliographie = OutilsXML.premierEnfant(parent, "BIBLIOGRAPHIE");
        if (el_bibliographie != null) {
            final ArrayList<Element> references = OutilsXML.enfants(el_bibliographie, "REFERENCE_BIBLIOGRAPHIQUE");
            for (Element el_ref : references)
                bibliographie.ajouterReference(el_ref);
        }
    }
    
    private void ajouterGlossaire(final Element parent) {
        final Element el_glossaire = OutilsXML.premierEnfant(parent, "GLOSSAIRE");
        if (el_glossaire != null) {
            final ArrayList<Element> definitions = OutilsXML.enfants(el_glossaire, "DEFINITION_GLOSSAIRE");
            for (Element el_def : definitions) {
                final String mot = el_def.getAttribute("mot");
                final String definition = OutilsXML.valeurElement(el_def);
                if (!"".equals(mot) && definition != null && !"".equals(definition))
                    glossaire.put(mot, definition);
            }
        }
    }
    
    private void ajouterReponsesPassees(final Element parent, final SuiviEDC suivi) {
        final ArrayList<Element> exercices = OutilsXML.enfants(parent, "EXERCICE");
        for (Element exercice : exercices) {
            final boolean reconsultable = !"non".equals(exercice.getAttribute("reconsultable"));
            if (reconsultable)
                return;
            final ArrayList<Element> enfants = OutilsXML.enfants(exercice);
            for (Element enfant : enfants) {
                if ("QUESTION_LIBRE".equals(enfant.getLocalName()) || "QUESTION_QUESTIONNAIRE".equals(enfant.getLocalName())) {
                    final Element question = enfant;
                    final String nom_variable = question.getAttribute("nom_variable");
                    if (suivi.variableDefinie(nom_variable)) {
                        final String valeur_variable = "" + suivi.getValeurVariable(nom_variable);
                        if ("QUESTION_LIBRE".equals(question.getLocalName()))
                            question.setAttributeNS(null, "valeur", valeur_variable);
                        else if ("QUESTION_QUESTIONNAIRE".equals(question.getLocalName()))
                            question.setAttributeNS(null, "points", valeur_variable);
                    }
                }
            }
        }
        final ArrayList<Element> blocs = OutilsXML.enfants(parent, "BLOC");
        for (Element bloc : blocs)
            ajouterReponsesPassees(bloc, suivi);
    }
    
    
    public String getLabel() {
        return(label);
    }
    
    public String getTitre() {
        return(titre);
    }
    
    public String labelPremiereEtape() {
        if (plan == null)
            return(null);
        return(plan.labelPremiereEtape());
    }
    
    /**
     * Renvoie le label de la premire section de l'tape prcise en paramtre
     */
    public String labelPremiereSection(final String label_etape) {
        final ArrayList<String> sections_etape = labels_sections.get(label_etape);
        if (sections_etape == null || sections_etape.isEmpty())
            return(null);
        return(sections_etape.get(0));
    }
    
    public String labelSectionSuivante(final String label_etape, final String label_section) {
        if (label_etape == null || label_section == null)
            return(null);
        final ArrayList<String> sections_etape = labels_sections.get(label_etape);
        if (sections_etape == null || sections_etape.isEmpty())
            return(null);
        boolean suivante = false;
        for (String label : sections_etape) {
            if (suivante)
                return(label);
            if (label_section.equals(label))
                suivante = true;
        }
        return(null);
    }
    
    public String labelSectionPrecedente(final String label_etape, final String label_section) {
        if (label_etape == null || label_section == null)
            return(null);
        final ArrayList<String> sections_etape = labels_sections.get(label_etape);
        if (sections_etape == null || sections_etape.isEmpty())
            return(null);
        String label_precedent = null;
        for (String label : sections_etape) {
            if (label_section.equals(label))
                return(label_precedent);
            label_precedent = label;
        }
        return(null);
    }
    
    public File getFichierContrib(final SuiviEDC suivi) {
        final String label_etape = suivi.getLabelEtape();
        final String label_section = suivi.getLabelSection();
        if (label_section != null && contribs_sections.get(label_section) != null)
            return(contribs_sections.get(label_section));
        return(contribs_etapes.get(label_etape));
    }
    
    public boolean etapeBloquante(final String label_etape) {
         return(etapes_bloquantes.contains(label_etape));
    }
    
    /**
     * Renvoie le label de l'tape qui contient la section dont le label est donn en paramtre.
     */
    public String labelEtapeSection(final String label_section) {
        return(etape_section.get(label_section));
    }
    
    /**
     * Renvoie l'lment DOM d'une tape  partir de son label
     */
    private Element lireEtape(final String label) throws ExceptionMICA {
        if (label == null)
            throw new ExceptionMICA("lireEtape: label null");
        final File fichier_contrib = contribs_etapes.get(label);
        if (fichier_contrib == null)
            throw new ExceptionMICA("Impossible de trouver l'tape " + label, true);
        final Document doc_contrib = OutilsXML.lectureDocumentXML(fichier_contrib);
        final Element racine = doc_contrib.getDocumentElement();
        final ArrayList<Element> etapes = OutilsXML.enfants(racine, "ETAPE");
        for (Element etape : etapes)
            if (label.equals(etape.getAttribute("label")))
                return(etape);
        return(null);
    }
    
    /**
     * Renvoie l'lment DOM d'une section  partir de son label
     */
    private Element lireSection(final String label) throws ExceptionMICA {
        if (label == null)
            throw new ExceptionMICA("lireSection: label null");
        final File fichier_contrib = contribs_sections.get(label);
        if (fichier_contrib == null)
            throw new ExceptionMICA("Impossible de trouver la section " + label, true);
        final Document doc_contrib = OutilsXML.lectureDocumentXML(fichier_contrib);
        final Element racine = doc_contrib.getDocumentElement();
        final ArrayList<Element> etapes = OutilsXML.enfants(racine, "ETAPE");
        for (Element etape : etapes) {
            final Element section = lireSection(etape, label);
            if (section != null)
                return(section);
        }
        return(null);
    }
    
    private Element lireSection(final Element parent, final String label) throws ExceptionMICA {
        final ArrayList<Element> sections = OutilsXML.enfants(parent, "SECTION");
        for (Element section : sections) {
            if (label.equals(section.getAttribute("label")))
                return(section);
            final Element ssection = lireSection(section, label);
            if (ssection != null)
                return(ssection);
        }
        return(null);
    }
    
    /**
     * Envoie sur la sortie donne en paramtre la page courante correspondant au suivi donn en paramtre.
     */
    public void afficherPage(final HttpServletResponse resp, final SuiviEDC suivi) throws ExceptionMICA {
        if (suivi.getLabelEtape() == null)
            throw new ExceptionMICA("afficherPage: aucune tape courante ?!?");
        final Element etape = lireEtape(suivi.getLabelEtape());
        final Element section;
        if (suivi.getLabelSection() == null)
            section = null;
        else
            section = lireSection(suivi.getLabelSection());
        final Document doc_inter = creationDocumentIntermediaire(suivi, etape, section);
        
        resp.setContentType("text/html; charset=ISO-8859-1");
        final OutputStream out;
        try {
            out = resp.getOutputStream();
        } catch (IOException ex) {
            throw new ExceptionMICA("afficherPage: HttpServletResponse.getOutputStream", ex);
        }
        OutilsXML.transformationXSLT(out, doc_inter, fichier_xsl);
    }
    
    /**
     * Cration du document envoy  la feuille de style affichage_MICA.xsl
     */
    private Document creationDocumentIntermediaire(final SuiviEDC suivi, final Element etape, final Element section) throws ExceptionMICA {
        final Document doc_inter = OutilsXML.nouveauDocumentDOM();
        final Element page = doc_inter.createElementNS(null, "PAGE_MICA");
        page.setAttributeNS(null, "titre_edc", titre);
        page.setAttributeNS(null, "label_etape", suivi.getLabelEtape());
        page.setAttributeNS(null, "titre_etape", etape.getAttribute("titre"));
        if (section != null) {
            page.setAttributeNS(null, "label_section", suivi.getLabelSection());
            page.setAttributeNS(null, "titre_section", section.getAttribute("titre"));
        }
        page.setAttributeNS(null, "bibliotheque", bibliotheque == null ? "non" : "oui");
        page.setAttributeNS(null, "bibliographie", bibliographie.vide() ? "non" : "oui");
        page.setAttributeNS(null, "glossaire", glossaire.isEmpty() ? "non" : "oui");
        if (nom_variable_progression != null) {
            final long valeur_progression = Math.round(getValeurVariable(nom_variable_progression, suivi));
            page.setAttributeNS(null, "progression", "" + valeur_progression);
        }
        if (plan != null) {
            final Element plan_edc = plan.elementInter(doc_inter, suivi);
            page.appendChild(plan_edc);
        }
        if (section != null) {
            final Element plan_etape = doc_inter.createElementNS(null, "PLAN_ETAPE");
            ajouterSectionsAuPlanEtape(plan_etape, etape, suivi.getLabelSection());
            page.appendChild(plan_etape);
            final Element liens_section = doc_inter.createElementNS(null, "LIENS_SECTION");
            final boolean page_suivante = (labelSectionSuivante(suivi.getLabelEtape(), suivi.getLabelSection()) != null);
            liens_section.setAttributeNS(null, "page_suivante", page_suivante ? "oui" : "non");
            final boolean page_precedente = (labelSectionPrecedente(suivi.getLabelEtape(), suivi.getLabelSection()) != null);
            liens_section.setAttributeNS(null, "page_precedente", page_precedente ? "oui" : "non");
            page.appendChild(liens_section);
        }
        
        final Element contenu = doc_inter.createElementNS(null, "CONTENU");
        if (section != null)
            remplacerVariables(contenu, section, suivi);
        else
            remplacerVariables(contenu, etape, suivi);
        
        ajouterDimensionsImages(contenu, getFichierContrib(suivi));
        
        ajouterReponsesPassees(contenu, suivi);
        
        page.appendChild(contenu);
        
        final ArrayList<Element> enchainements_etape = OutilsXML.enfants(etape, "ENCHAINEMENT");
        if (enchainements_etape.size() > 0) {
            final ArrayList<Element> enchainements_possibles = new ArrayList<Element>();
            for (Element enchainement_etape : enchainements_etape) {
                final String condition = enchainement_etape.getAttribute("condition");
                if ("".equals(condition) || testCondition(condition, suivi)) {
                    final Element enchainement_possible = doc_inter.createElementNS(null, "ENCHAINEMENT");
                    final String label_etape = enchainement_etape.getAttribute("etape");
                    enchainement_possible.setAttributeNS(null, "label_etape", label_etape);
                    if (!"".equals(enchainement_etape.getAttribute("titre")))
                        enchainement_possible.setAttributeNS(null, "titre", enchainement_etape.getAttribute("titre"));
                    else
                        enchainement_possible.setAttributeNS(null, "titre", titres_etapes.get(label_etape));
                    enchainements_possibles.add(enchainement_possible);
                }
            }
            if (enchainements_possibles.size() > 0) {
                final Element enchainements = doc_inter.createElementNS(null, "ENCHAINEMENTS");
                for (Element enchainement_possible : enchainements_possibles)
                    enchainements.appendChild(enchainement_possible);
                page.appendChild(enchainements);
            }
        }
        doc_inter.appendChild(page);
        return(doc_inter);
    }
    
    private void ajouterSectionsAuPlanEtape(final Element plan_etape, final Element parent, final String label_section_active) {
        final Document doc_inter = plan_etape.getOwnerDocument();
        final ArrayList<Element> enfants = OutilsXML.enfants(parent);
        for (Element el : enfants) {
            if ("SECTION".equals(el.getLocalName())) {
                final Element section = doc_inter.createElementNS(null, "SECTION");
                final String label_section = el.getAttribute("label");
                section.setAttributeNS(null, "label", el.getAttribute("label"));
                section.setAttributeNS(null, "titre", el.getAttribute("titre"));
                section.setAttributeNS(null, "active", label_section.equals(label_section_active) ? "oui" : "non");
                ajouterSectionsAuPlanEtape(section, el, label_section_active);
                plan_etape.appendChild(section);
            }
        }
    }
    
    private void remplacerVariables(final Element contenu, final Element parent, final SuiviEDC suivi) {
        final Document doc_inter = contenu.getOwnerDocument();
        for (Node n=parent.getFirstChild(); n != null; n = n.getNextSibling()) {
            if (n instanceof Element) {
                final Element sousel = (Element)n;
                final String tag = sousel.getLocalName();
                if ("SECTION".equals(tag)) {
                    final Element section = doc_inter.createElementNS(null, "SECTION");
                    remplacerVariables(section, sousel, suivi);
                    contenu.appendChild(section);
                } else if ("REF_SECTION".equals(tag))
                    ; // pb
                else if ("VALEUR_VARIABLE".equals(tag)) {
                    final String nom = sousel.getAttribute("nom");
                    final Double valeur = new Double(getValeurVariable(nom, suivi));
                    final Node nval;
                    if (valeur == null)
                        nval = doc_inter.createTextNode("[variable sans valeur: " + nom + "]");
                    else
                        nval = doc_inter.createTextNode(valeur.toString());
                    contenu.appendChild(nval);
                } else if (!"BIBLIOTHEQUE".equals(tag) && !"BIBLIOGRAPHIE".equals(tag) && !"GLOSSAIRE".equals(tag) && !"ENCHAINEMENT".equals(tag)) {
                    final Element nouveau = (Element)doc_inter.importNode(sousel, false);
                    contenu.appendChild(nouveau);
                    remplacerVariables(nouveau, sousel, suivi);
                }
            } else
                contenu.appendChild(doc_inter.importNode(n, true));
        }
    }
    
    private void ajouterDimensionsImages(final Element parent, final File fichier_contrib) throws ExceptionMICA {
        for (Node n=parent.getFirstChild(); n != null; n = n.getNextSibling()) {
            if (n instanceof Element) {
                final Element sousel = (Element)n;
                final String tag = sousel.getLocalName();
                if ("ZONE_IMAGE".equals(tag)) {
                    final String localisation = sousel.getAttribute("localisation");
                    final ArrayList<Element> fichiers = OutilsXML.enfants(sousel, "FICHIER");
                    for (Element el_fichier : fichiers) {
                        final String nom = el_fichier.getAttribute("nom");
                        final File fichier = new File(fichier_contrib.getParentFile(), nom);
                        final Dimensions dim = new Dimensions(fichier);
                        final int largeur1 = dim.getLargeur();
                        final int hauteur1 = dim.getHauteur();
                        el_fichier.setAttributeNS(null, "largeur1", ""+largeur1);
                        el_fichier.setAttributeNS(null, "hauteur1", ""+hauteur1);
                        boolean redim;
                        int largeur_max, hauteur_max;
                        if ("page".equals(localisation) && (largeur1 > 300 || hauteur1 > 300)) {
                            redim = true;
                            largeur_max = 300;
                            hauteur_max = 300;
                        } else if ("icne".equals(localisation) && (largeur1 > 100 || hauteur1 > 100)) {
                            redim = true;
                            largeur_max = 100;
                            hauteur_max = 100;
                        } else {
                            redim = false;
                            largeur_max = 0;
                            hauteur_max = 0;
                        }
                        el_fichier.setAttributeNS(null, "redim", redim ? "oui" : "non");
                        if (redim) {
                            int largeur2, hauteur2;
                            // en fait pour l'instant on laisse le navigateur de l'utilisateur redimensionner les images et animations
                            if ((1.0*largeur1)/largeur_max > (1.0*hauteur1)/hauteur_max) {
                                largeur2 = largeur_max;
                                hauteur2 = (int)Math.round(((1.0*largeur_max)/largeur1) * hauteur1);
                                if (hauteur2 > hauteur_max)
                                    hauteur2 = hauteur_max;
                            } else {
                                hauteur2 = hauteur_max;
                                largeur2 = (int)Math.round(((1.0*hauteur_max)/hauteur1) * largeur1);
                                if (largeur2 > largeur_max)
                                    largeur2 = largeur_max;
                            }
                            el_fichier.setAttributeNS(null, "largeur2", ""+largeur2);
                            el_fichier.setAttributeNS(null, "hauteur2", ""+hauteur2);
                        } else {
                            el_fichier.setAttributeNS(null, "largeur2", ""+largeur1);
                            el_fichier.setAttributeNS(null, "hauteur2", ""+hauteur1);
                        }
                    }
                } else
                    ajouterDimensionsImages(sousel, fichier_contrib);
            }
        }
    }
    
    protected double getValeurVariable(final String nom, final SuiviEDC suivi) {
        if (formules_variables.get(nom) == null)
            return(suivi.getValeurVariable(nom));
        
        final String formule = formules_variables.get(nom);
        return((new Formule(formule, this, suivi)).evaluer());
    }
    
    public boolean testCondition(final String condition, final SuiviEDC suivi) {
        double res = (new Formule(condition, this, suivi)).evaluer();
        return(res == 1);
    }
    
    // Renvoie true si un enchanement est possible entre etape1 et etape2, avec le suivi pass en paramtre.
    // Le suivi n'est pas pris en compte s'il est null.
    public boolean enchainementPossible(final String label_etape1, final String label_etape2, final SuiviEDC suivi) throws ExceptionMICA {
        // remarque: on pourrait optimiser a en gardant les enchanements en mmoire
        if (label_etape2 == null)
            throw new ExceptionMICA("enchainementPossible: label_etape2 null");
        final Element etape = lireEtape(label_etape1);
        if (etape == null)
            throw new ExceptionMICA("enchainementPossible: Etape " + label_etape1 + " introuvable !");
        final ArrayList<Element> enchainements_etape = OutilsXML.enfants(etape, "ENCHAINEMENT");
        for (Element enchainement_etape : enchainements_etape) {
            final String label_etape_enchainement = enchainement_etape.getAttribute("etape");
            if (suivi == null) {
                if (label_etape2.equals(label_etape_enchainement))
                    return(true);
            } else if (label_etape2.equals(label_etape_enchainement)) {
                final String condition = enchainement_etape.getAttribute("condition");
                if ("".equals(condition) || testCondition(condition, suivi))
                    return(true);
            }
        }
        return(false);
    }
    
    
    /**
     * Vrifie que l'tape n'est pas bloquante ou que toutes les rponses aux exercices ont t envoyes,
     * et qu'elles n'ont pas dj t envoyes si l'exercice n'est pas reconsultable.
     */
    public void verifierParametres(final Hashtable<String, ArrayList<String>> parametres, final SuiviEDC suivi) throws ExceptionMICA {
        final String label_etape = suivi.getLabelEtape();
        if (label_etape == null)
            throw new ExceptionMICA("verifierParametres: aucune tape courante ?!?");
        final boolean bloquante = etapeBloquante(label_etape);
        final Element etape = lireEtape(label_etape);
        final boolean etape_passee = (suivi.getValeurVariable(label_etape) == 1);
        final ArrayList<Element> exercices = OutilsXML.enfants(etape, "EXERCICE");
        for (int i=0; i<exercices.size(); i++) {
            final Element exercice = exercices.get(i);
            final boolean reconsultable = !"non".equals(exercice.getAttribute("reconsultable"));
            Node item = exercice.getFirstChild();
            int numero_question = 0;
            while (item != null) {
                if (item instanceof Element) {
                    final Element sousel = (Element)item;
                    if ("QUESTION_LIBRE".equals(sousel.getLocalName()) || "QUESTION_QUESTIONNAIRE".equals(sousel.getLocalName())) {
                        numero_question++;
                        final String nom_parametre = "exercice" + (i+1) + "_question" + numero_question;
                        final ArrayList<String> valeurs = parametres.get(nom_parametre);
                        if (valeurs == null || valeurs.isEmpty() || "".equals(valeurs.get(0))) {
                            if (!etape_passee && bloquante)
                                throw new ExceptionMICA("L'tape est bloquante: il faut rpondre aux exercices pour continuer", true);
                        } else if (etape_passee && !reconsultable)
                            throw new ExceptionMICA("Vous n'avez pas le droit d'envoyer de nouvelles valeurs pour cet exercice.", true);
                    }
                }
                item = item.getNextSibling();
            }
        }
        return;
    }
    
    
    /**
     * Mise  jour des variables du suivi  partir des rponses aux exercices
     */
    public void majVariables(final Hashtable<String, ArrayList<String>> parametres, final SuiviEDC suivi) throws ExceptionMICA {
        if (suivi.getLabelEtape() == null)
            throw new ExceptionMICA("majVariables: aucune tape courante ?!?");
        for (Map.Entry<String, ArrayList<String>> parametre : parametres.entrySet()) {
            final String nom_parametre = parametre.getKey();
            final ArrayList<String> valeurs_parametre = parametre.getValue();
            // nom_parametre : exercice{numro exo dans la page}_question{numro question dans l'exo}
            final int pu1 = nom_parametre.indexOf('_');
            if (pu1 == -1)
                throw new ExceptionMICA("Nom de paramtre incorrect: " + nom_parametre);
            final String s_numero_exo = nom_parametre.substring("exercice".length(), pu1);
            final String s_numero_question = nom_parametre.substring(pu1 + 1 + "question".length());
            int numero_exo;
            int numero_question;
            try {
                numero_exo = Integer.valueOf(s_numero_exo).intValue();
                numero_question = Integer.valueOf(s_numero_question).intValue();
            } catch (NumberFormatException ex) {
                throw new ExceptionMICA("majVariables " + nom_parametre, ex);
            }
            final Element etape = lireEtape(suivi.getLabelEtape());
            final Element question = chercherQuestion(etape, numero_exo, numero_question);
            final String nom_variable = question.getAttribute("nom_variable");
            if ("".equals(nom_variable))
                throw new ExceptionMICA("Aucune variable dfinie pour la question " + numero_question + " de l'exercice " + numero_exo);
            if ("QUESTION_LIBRE".equals(question.getLocalName())) {
                if (valeurs_parametre.size() > 1)
                    throw new ExceptionMICA("Plus d'une valeur pour une rponse libre ?!?");
                if (!"".equals(valeurs_parametre.get(0))) {
                    try {
                        suivi.setValeurVariable(nom_variable, Double.valueOf(valeurs_parametre.get(0)));
                    } catch (NumberFormatException ex) {
                        throw new ExceptionMICA(valeurs_parametre.get(0) + " n'est pas un nombre !", true);
                    }
                }
            } else {
                final String type = question.getAttribute("type");
                if ("QCU".equals(type)) {
                    if (valeurs_parametre.size() > 1)
                        throw new ExceptionMICA("Plus d'une valeur pour une rponse  une question  choix unique ?!?");
                    final String s_numero_reponse = valeurs_parametre.get(0);
                    final int numero_reponse;
                    try {
                        numero_reponse = Integer.valueOf(s_numero_reponse).intValue();
                    } catch (NumberFormatException ex) {
                        throw new ExceptionMICA("majVariables " + nom_parametre, ex);
                    }
                    final ArrayList<Element> reponses = OutilsXML.enfants(question, "REPONSE_QUESTIONNAIRE");
                    if (numero_reponse < 1 || numero_reponse > reponses.size())
                        throw new ExceptionMICA("Mauvais numro de rponse : " + numero_reponse);
                    final Element reponse = reponses.get(numero_reponse - 1);
                    final String s_points = reponse.getAttribute("points");
                    try {
                        suivi.setValeurVariable(nom_variable, Double.valueOf(s_points));
                    } catch (NumberFormatException ex) {
                        throw new ExceptionMICA(s_points + " n'est pas un nombre !", true);
                    }
                } else {
                    double points = 0;
                    for (String valeur_parametre : valeurs_parametre) {
                        final int numero_reponse;
                        try {
                            numero_reponse = Integer.valueOf(valeur_parametre).intValue();
                        } catch (NumberFormatException ex) {
                            throw new ExceptionMICA("majVariables " + nom_parametre, ex);
                        }
                        final ArrayList<Element> reponses = OutilsXML.enfants(question, "REPONSE_QUESTIONNAIRE");
                        if (numero_reponse < 1 || numero_reponse > reponses.size())
                            throw new ExceptionMICA("Mauvais numro de rponse : " + numero_reponse);
                        final Element reponse = reponses.get(numero_reponse - 1);
                        final String s_points = reponse.getAttribute("points");
                        final double points_reponse;
                        try {
                            points_reponse = Double.valueOf(s_points).doubleValue();
                        } catch (NumberFormatException ex) {
                            throw new ExceptionMICA("majVariables: Le nombre de points d'une rponse n'est pas un nombre: " + s_points, ex, true);
                        }
                        points += points_reponse;
                    }
                    suivi.setValeurVariable(nom_variable, new Double(points));
                }
            }
        }
    }
    
    
    /**
     * Mise  jour du suivi aprs un changement d'tape ou de section.
     */
    public void majSuivi(final SuiviEDC suivi, final String label_nouvelle_etape, final String label_nouvelle_section) throws ExceptionMICA {
        suivi.etapeTerminee();
        if (label_nouvelle_etape != null) {
            suivi.setLabelEtape(label_nouvelle_etape);
            if (label_nouvelle_section == null)
                suivi.setLabelSection(labelPremiereSection(label_nouvelle_etape));
            final Element etape = lireEtape(label_nouvelle_etape);
            final ArrayList<Element> enchainements_etape = OutilsXML.enfants(etape, "ENCHAINEMENT");
            if (enchainements_etape.size() == 0)
                suivi.setTerminee();
        }
        if (label_nouvelle_section != null)
            suivi.setLabelSection(label_nouvelle_section);
    }
    
    private Element chercherQuestion(final Element etape, final int numero_exo, final int numero_question) throws ExceptionMICA {
        final ArrayList<Element> exercices = OutilsXML.enfants(etape, "EXERCICE");
        if (numero_exo < 1 || numero_exo > exercices.size())
            throw new ExceptionMICA("Mauvais numro d'exercice : " + numero_exo);
        final Element exercice = exercices.get(numero_exo - 1);
        final ArrayList<Element> questions = new ArrayList<Element>();
        Node item = exercice.getFirstChild();
        while (item != null) {
            if (item instanceof Element) {
                final Element sousel = (Element)item;
                if ("QUESTION_LIBRE".equals(sousel.getLocalName()) || "QUESTION_QUESTIONNAIRE".equals(sousel.getLocalName()))
                    questions.add(sousel);
            }
            item = item.getNextSibling();
        }
        if (numero_question < 1 || numero_question > questions.size())
            throw new ExceptionMICA("Mauvais numro de question : " + numero_question);
        final Element question = questions.get(numero_question - 1);
        return(question);
    }
    
    
    public void afficherBibliotheque(final HttpServletResponse resp) throws ExceptionMICA {
        if (bibliotheque == null)
            throw new ExceptionMICA("Aucune bibliothque  afficher !");
        //resp.setContentType("text/html; charset=ISO-8859-1"); fait dans ServletMICA
        final OutputStream out;
        try {
            out = resp.getOutputStream();
        } catch (IOException ex) {
            throw new ExceptionMICA("afficherBibliotheque: HttpServletResponse.getOutputStream", ex);
        }
        OutilsXML.transformationXSLT(out, bibliotheque, fichier_xsl);
    }
    
    public void afficherSommaire(final PrintWriter pw, final SuiviEDC suivi) {
        ServletMICA.entete(pw, titre, "Sommaire");
        
        plan.afficherSommaire(pw, suivi);
        pw.println("<p><a href=\"MICA?page=accueil\">Revenir  l'accueil</a></p>");
        
        ServletMICA.pied(pw);
    }
    
    
    public void afficherGlossaire(final PrintWriter pw) {
        ServletMICA.entete(pw, titre, "Glossaire");
        
        final String lettres = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        pw.println("<p>");
        for (int i=0; i<lettres.length(); i++) {
            final char c = lettres.charAt(i);
            pw.println("[<a href=\"#" + Character.toLowerCase(c) + "\">" + c + "</a>] ");
        }
        pw.println("</p>");
        
        final String t1 = "abcdefghijklmnopqrstuvwxyz";
        final String t2 = "AABCCDEEEEFGHIIIJKLMNOOPQRSTUUVWXYZ";
        final String t3 = " ";
        final String t4 = "_aaeeeiioouu";
        for (int i=0; i<lettres.length(); i++) {
            final char c = lettres.charAt(i);
            final ArrayList<String> selection_mots = new ArrayList<String>();
            for (Enumeration<String> mots = glossaire.keys(); mots.hasMoreElements(); ) {
                final String mot = mots.nextElement();
                char lettre1 = mot.charAt(0);
                final int ind = t1.indexOf(lettre1);
                if (ind != -1)
                    lettre1 = t2.charAt(ind);
                if (lettre1 == c)
                    selection_mots.add(mot);
            }
            if (!selection_mots.isEmpty()) {
                pw.println("<a name=\"" + Character.toLowerCase(c) + "\"></a>");
                pw.println("<h2>- " + c + " -</h2>");
                pw.println("<dl>");
                for (String mot : selection_mots) {
                    String ancre = mot;
                    for (int j=0; j<t3.length(); j++)
                        ancre = ancre.replace(t3.charAt(j), t4.charAt(j));
                    pw.println("<dt><a name=\"" + ancre + "\"><b>" + mot + "</b></a></dt>");
                    pw.println("<dd>" + OutilsXML.encoder(glossaire.get(mot)) + "</dd>");
                }
                pw.println("</dl>");
            }
        }
        
        ServletMICA.pied(pw);
    }
    
    public void afficherBibliographie(final PrintWriter pw) {
        ServletMICA.entete(pw, titre, "Bibliographie");
        
        bibliographie.afficher(pw);
        
        ServletMICA.pied(pw);
    }
    
    
    /**
     * Envoie sur la sortie donne en paramtre l'quation ayant le numro donn dans le
     * contenu de la page ou de la section courante
     */
    public void envoyerEquation(final HttpServletResponse resp, final int numero, final SuiviEDC suivi) throws ExceptionMICA {
        if (suivi.getLabelEtape() == null)
            throw new ExceptionMICA("envoyerEquation: aucune tape courante ?!?");
        final Element parent;
        if (suivi.getLabelSection() == null)
            parent = lireEtape(suivi.getLabelEtape());
        else
            parent = lireSection(suivi.getLabelSection());
        final Element el_equation = chercherEquation(parent, numero);
        if (el_equation == null)
            throw new ExceptionMICA("Etape " + suivi.getLabelEtape() + ": quation introuvable: " + numero, true);
        final String imageBase64 = OutilsXML.valeurElement(el_equation);
        final DecodeurBase64 decodeur = new DecodeurBase64(new StringReader(imageBase64));
        resp.setContentType("image/png");
        //resp.setContentLength(?);
        final OutputStream out;
        try {
            out = resp.getOutputStream();
        } catch (IOException ex) {
            throw new ExceptionMICA("envoyerEquation: HttpServletResponse.getOutputStream", ex);
        }
        int c;
        try {
            while ((c=decodeur.read()) != -1)
                out.write(c);
            decodeur.close();
            out.close();
        } catch (IOException ex) {
            throw new ExceptionMICA("envoyerEquation " + numero, ex);
        }
    }
    
    private Element chercherEquation(final Element parent, final int numero) {
        int i = 0;
        Node n = parent.getFirstChild();
        while (n != null) {
            if (n instanceof Element) {
                final Element sousel = (Element)n;
                final String tag = sousel.getLocalName();
                if ("EQUATION".equals(tag)) {
                    i++;
                    if (i == numero)
                        return(sousel);
                }
            }
            if (n.getFirstChild() != null)
                n = n.getFirstChild();
            else {
                while (n != null && n.getNextSibling() == null) {
                    n = n.getParentNode();
                    if (n == parent)
                        n = null;
                }
                if (n != null)
                    n = n.getNextSibling();
            }
        }
        return(null);
    }
    
}
