/*
Mex - Mini Editeur XML

Copyright (C) 2004 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package mex;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.text.BadLocationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ErrorHandler;
import org.apache.xerces.parsers.DOMParser;

/**
 * Fentre de validation du document, affichant une liste d'erreurs. La validation utilise Xerces.
 * Un clic sur une erreur slectionne la ligne correspondante dans le fichier source.
 */
public class FenetreValidation extends JFrame implements MouseListener, ActionListener {

    File fichierXML;
    File fichierSchema;
    JEditorPane editorPane;
    String contenu;
    ArrayList erreurs;
    FenetreSource sourceFrame;

    /**
     * Constructeur pour un document dont le schma n'est pas spcifi
     * (un schma ou DTD doit etre prcis dans le document)
     */
    public FenetreValidation(File fichierXML, FenetreSource sourceFrame) {
        super("Liste d'erreurs pour " + fichierXML.getName());
        this.sourceFrame = sourceFrame;
        newdoc(fichierXML, null);
    }
    
    /**
     * Constructeur pour un document dont on spcifie le schma extrieur
     */
    public FenetreValidation(File fichierXML, File fichierSchema, FenetreSource sourceFrame) {
        super("Liste d'erreurs pour " + fichierXML.getName());
        this.sourceFrame = sourceFrame;
        newdoc(fichierXML, fichierSchema);
    }
    
    /**
     * Ouverture d'un document et validation. fichierSchema peut tre null.
     */
    public void newdoc(File fichierXML, File fichierSchema) {
        this.fichierXML = fichierXML;
        this.fichierSchema = fichierSchema;
        Rectangle fr = sourceFrame.getBounds();
        setLocation(fr.x + fr.width/2, fr.y + fr.height/2);
        setSize(new Dimension(620, 460));
        affichage();
        miseAJour();
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                try {
                    editorPane.scrollRectToVisible(editorPane.modelToView(0));
                } catch (BadLocationException ex) {
                }
            }
        });
    }
    
    protected void affichage() {
        editorPane = new JEditorPane();
        editorPane.setEditable(false);
        JScrollPane paneScrollPane = new JScrollPane(editorPane);
        paneScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        paneScrollPane.setPreferredSize(new Dimension(600, 400));
        
        /*JPanel boutonsP = new JPanel();
        boutonsP.setLayout(new FlowLayout());
        JButton boutonMAJ = new JButton("Mise  jour");
        boutonMAJ.addActionListener(this);
        boutonsP.add(boutonMAJ);
        */
        JPanel contentPane = new JPanel(new BorderLayout());
        contentPane.add(paneScrollPane, BorderLayout.CENTER);
        //contentPane.add(boutonsP, BorderLayout.NORTH);
        setContentPane(contentPane);
        
        editorPane.addMouseListener(this);
    }
    
    protected void print(String s) {
        contenu += s;
    }
    
    protected void println(String s) {
        contenu += s + "\n";
    }
    
    protected String getTargetNamespace() {
        try {
            DOMParser parser = new DOMParser();
            String furl = fichierSchema.toURL().toString(); // to fix problem with xerces and 
            parser.parse(furl);
            Element racine = parser.getDocument().getDocumentElement();
            if (!"schema".equals(racine.getLocalName()))
                System.err.println("la racine du schma n'est pas schema ?!?");
            String target = racine.getAttribute("targetNamespace");
            if ("".equals(target))
                return(null);
            else
                return(target);
        } catch (Exception e) {
            e.printStackTrace(System.err);
            return(null);
        }
    }
    
    /**
     * Mise  jour de la liste d'erreurs
     */
    public void miseAJour() {
        String nomFichier = fichierXML.getPath();
        String nomSchema = null;
        if (fichierSchema != null)
            nomSchema = fichierSchema.getAbsolutePath();
        
        contenu = "";
        erreurs = new ArrayList();
        boolean[] terreur = new boolean[1];
        try {
            DOMParser parser = new DOMParser();
            
            terreur[0] = false;
            ErrorStorer ef = new ErrorStorer(terreur);
            parser.setErrorHandler(ef);
            
            parser.setFeature("http://apache.org/xml/features/allow-java-encodings", true);
            parser.setFeature("http://xml.org/sax/features/validation", true);
            
            parser.setFeature("http://apache.org/xml/features/continue-after-fatal-error", true);
            
            if (fichierSchema != null) {
                while (nomSchema.indexOf(' ') != -1) {
                    int is = nomSchema.indexOf(' ');
                    nomSchema = nomSchema.substring(0, is) + "%20" + nomSchema.substring(is + 1);
                }
                String targetNamespace = getTargetNamespace();
                if (targetNamespace == null || "".equals(targetNamespace)) {
                    parser.setProperty("http://apache.org/xml/properties/schema/external-schemaLocation",
                        "anyURI " + nomSchema);
                    parser.setProperty("http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation",
                        nomSchema);
                } else {
                    parser.setProperty("http://apache.org/xml/properties/schema/external-schemaLocation",
                        targetNamespace + " " + nomSchema);
                }
            }
            parser.parse(nomFichier);
            Document domdoc = parser.getDocument();
        } catch (Exception e) {
            e.printStackTrace(System.err);
            return;
        }
        if (!terreur[0])
            print("Ce fichier est parfait !");
        editorPane.setText(contenu);
        setVisible(true);
    }
    
    class ErrorStorer implements ErrorHandler {
        boolean terreur[];
        
        public ErrorStorer(boolean terreur[]) {
            this.terreur = terreur;
        }

        public void warning(SAXParseException ex) {
            handleError(ex, "Warning");
        }

        public void error(SAXParseException ex) {
            handleError(ex, "Error");
        }

        public void fatalError(SAXParseException ex) throws SAXException {
            handleError(ex, "Fatal error");
        }
        
        private void handleError(SAXParseException ex, String terr) {
            erreurs.add(new Erreur(ex.getLineNumber(), contenu.length()));
            terreur[0] = true;
            println(terr + "  at line number " + ex.getLineNumber() + ": ");
            println(ex.getMessage() + "\n");
        }
    }
    
    public void actionPerformed(ActionEvent e) {
        miseAJour();
    }
    
    public void mouseClicked(MouseEvent e) {
        int pos = editorPane.viewToModel(e.getPoint());
        int p1 = 0;
        int p2 = 0;
        int ligne = 0;
        for (int i=0; i<erreurs.size(); i++) {
            Erreur err = (Erreur)erreurs.get(i);
            if (err.p1 > pos) {
                p2 = err.p1;
                break;
            } else {
                p1 = err.p1;
                ligne = err.ligne;
            }
        }
        if (p2 == 0)
            p2 = contenu.length();
        editorPane.setCaretPosition(p1);
        editorPane.moveCaretPosition(p2);
        
        sourceFrame.selectLigne(ligne);
        sourceFrame.toFront();
    }
    
    public void mouseEntered(MouseEvent e) {
    }
    
    public void mouseExited(MouseEvent e) {
    }
    
    public void mousePressed(MouseEvent e) {
    }
    
    public void mouseReleased(MouseEvent e) {
    }
    
    class Erreur {
        public int ligne;
        public int p1;
        public Erreur(int ligne, int p1) {
            this.ligne = ligne;
            this.p1 = p1;
        }
    }
}
