/*
Mex - Mini Editeur XML

Copyright (C) 2004 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package mex;

import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.AbstractDocument;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.Segment;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleContext;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.TabSet;
import javax.swing.text.TabStop;
import javax.swing.undo.*;

import org.w3c.dom.Document;

/**
 * Document Mex, permettant l'dition d'un document XML
 */
public class FenetreSource extends JFrame implements ActionListener {

    private File fichierXML;
    private JTextPane zoneTexte;
    private StyledDocument doc;
    private String encodage = "ISO-8859-1"; // encodage par dfaut
    private FenetreValidation fvalidation;
    private boolean modif;
    private UndoManager undoMgr;
    private Style styleElement;
    private Style styleNomAttribut;
    private Style styleValeurAttribut;
    private Style styleTexte;
    private Style styleEntite;
    private Style styleCommentaire;
    private int poscol, fincol;
    private boolean ignorerModifs;
    private JMenuItem menuAnnuler, menuRetablir;

    public FenetreSource() {
        super("Sans Titre");
        initFenetre();
        fichierXML = null;
        nouveau();
    }
    
    public FenetreSource(File fichierXML) {
        initFenetre();
        ouvrir(fichierXML);
    }
    
    public void nouveau() {
        fichierXML = null;
        zoneTexte.setText("");
        /*SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                zoneTexte.setText("<?xml version=\"1.0\" encoding=\"" + encodage + "\"?>\n");
                modif = false;
            }
        });*/
        modif = false;
        ignorerModifs = false;
        setVisible(true);
        zoneTexte.requestFocus();
    }
    
    protected void changementDoc() {
        boolean vignore = ignorerModifs;
        ignorerModifs = true;
        undoMgr = new UndoManager();
        doc.addUndoableEditListener(new MonEcouteurDAnnulations());
        
        doc.addDocumentListener(new DocumentListener() {
            public void insertUpdate(DocumentEvent e) {
                if (!ignorerModifs)
                    majCouleurs(e.getOffset(), e.getLength());
                if (e.getLength() == 1) {
                    try {
                        int off = e.getOffset();
                        String sins = zoneTexte.getText(off, 1);
                        if ("\n".equals(sins))
                            SwingUtilities.invokeLater(new AjoutIndentation(off));
                    } catch (BadLocationException ex) {
                        System.err.println("BadLocationException: " + ex.getMessage());
                    }
                }
                modif = true;
            }
            public void removeUpdate(DocumentEvent e) {
                if (!ignorerModifs)
                    majCouleurs(e.getOffset(), e.getLength());
                modif = true;
            }
            public void changedUpdate(DocumentEvent e) {
                if (!ignorerModifs)
                    majCouleurs(e.getOffset(), e.getLength());
                modif = true;
            }
        });
        
        // Monaco font looks much better than the default Courier on MacOS X with Java 1.4.1
        String[] fontnames = GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
        boolean trouv = false;
        for (int i=0; i<fontnames.length; i++)
            if ("Monaco".equals(fontnames[i])) {
                trouv = true;
                break;
            }
        if (trouv) {
            Style defaultStyle = zoneTexte.getStyle(StyleContext.DEFAULT_STYLE);
            StyleConstants.setFontFamily(defaultStyle, "Monaco");
            StyleConstants.setFontSize(defaultStyle, 12);
            zoneTexte.setFont(new Font("Monaco", Font.PLAIN, 12));
        }
        
        setTabs(4);
        
        ignorerModifs = vignore;
    }
    
    class AjoutIndentation implements Runnable {
        int pos;
        public AjoutIndentation(int pos) {
            this.pos = pos;
        }
        public void run() {
            Element para = doc.getParagraphElement(pos - 1);
            int debut = para.getStartOffset();
            int fin = para.getEndOffset();
            try {
                String ligne = zoneTexte.getText(debut, fin-debut);
                if (ligne.length() > 0 && ligne.charAt(ligne.length()-1) == '\n')
                    ligne = ligne.substring(0, ligne.length()-1);
                ligne += '*';
                int ind = ligne.indexOf(ligne.trim());
                if (ind > 0)
                    doc.insertString(pos+1, ligne.substring(0, ind), null);
            } catch (BadLocationException ex) {
                System.err.println("BadLocationException: " + ex.getMessage());
            }
        }
    }
    
    protected void initFenetre() {
        setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        zoneTexte = new JTextPane(); // disabled horizontal scrolling doesn't work with JEditorPane
        
        doc = zoneTexte.getStyledDocument();
        changementDoc();
        
        //zoneTexte.setFont(new Font("Monospaced", Font.PLAIN, 12));
        // setFont doesn't work for JTextPane
        
        Style defaultStyle = zoneTexte.getStyle(StyleContext.DEFAULT_STYLE);
        styleTexte = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleTexte, Color.black); // noir
        styleElement = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleElement, new Color(150, 0, 0)); // rouge fonc
        styleNomAttribut = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleNomAttribut, new Color(0, 0, 150)); // bleu fonc
        styleValeurAttribut = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleValeurAttribut, new Color(0, 100, 0)); // vert fonc
        styleEntite = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleEntite, new Color(0, 100, 100)); // cyan fonc
        styleCommentaire = zoneTexte.addStyle(null, defaultStyle);
        StyleConstants.setForeground(styleCommentaire, Color.gray); // gris
        
        JScrollPane paneScrollPane = new JScrollPane(zoneTexte);
        paneScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        
        JPanel boutonsP = new JPanel();
        boutonsP.setLayout(new FlowLayout());
        JButton boutonMAJ = new JButton("Valider");
        boutonMAJ.setActionCommand("valider");
        int cmdMenu = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
        zoneTexte.getInputMap().put(KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_K, cmdMenu), "valider");
        zoneTexte.getActionMap().put("valider", new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                valider();
            }
        });
        zoneTexte.getInputMap().put(KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_TAB, 0), "tabulation");
        zoneTexte.getInputMap().put(KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_TAB, Event.SHIFT_MASK), "shifttab");
        zoneTexte.getActionMap().put("tabulation", new ActionTab(false));
        zoneTexte.getActionMap().put("shifttab", new ActionTab(true));
        boutonMAJ.addActionListener(this);
        boutonsP.add(boutonMAJ);
        
        JPanel contentPane = new JPanel(new BorderLayout());
        contentPane.add(paneScrollPane, BorderLayout.CENTER);
        contentPane.add(boutonsP, BorderLayout.SOUTH);
        setContentPane(contentPane);
        
        initMenus();
        
        addWindowListener( new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                Mex.fermer(FenetreSource.this);
            }
        });
        
        int nbfen = Mex.nombreFenetres();
        setLocation(50+nbfen*20, 50+nbfen*20);
        Dimension dim = new Dimension(600, 700);
        Dimension ecran = getToolkit().getScreenSize();
        if (ecran.height < dim.height + 100)
            dim.height = ecran.height - 100;
        if (ecran.width < dim.width + 100)
            dim.width = ecran.width - 100;
        setSize(dim);
    }
    
    class ActionTab extends AbstractAction {
        boolean shift;
        public ActionTab(boolean shift) {
            super();
            this.shift = shift;
        }
        public void actionPerformed(ActionEvent e) {
            int start = zoneTexte.getSelectionStart();
            int end = zoneTexte.getSelectionEnd();
            String s = null;
            try {
                if (Math.abs(end-start) > 0) {
                    if (end < start) {
                        int tmp = end;
                        end = start;
                        start = tmp;
                    }
                    s = zoneTexte.getText(start, end-start);
                }
            } catch (BadLocationException ex) {
                System.err.println("BadLocationException: " + ex.getMessage());
            }
            String stab;
            if (s == null || s.indexOf('\n') == -1 || s.indexOf('\t') != -1 || s.length() < 1 || s.charAt(0) != ' ')
                stab = "\t";
            else
                stab = "    ";
            if (s != null && s.indexOf('\n') != -1) {
                if (!shift) {
                    s = stab + s;
                    int i = 0;
                    while (i < s.length()-1) {
                        if (s.charAt(i) == '\n') {
                            s = s.substring(0,i+1) + stab + s.substring(i+1);
                            i += stab.length();
                        }
                        i++;
                    }
                } else {
                    if (s.startsWith(stab))
                        s = s.substring(stab.length());
                    int i = 0;
                    while (i < s.length()-1-stab.length()) {
                        if (s.charAt(i) == '\n') {
                            if (s.substring(i+1).startsWith(stab))
                                s = s.substring(0,i+1) + s.substring(i+1+stab.length());
                        }
                        i++;
                    }
                }
                zoneTexte.replaceSelection(s);
                zoneTexte.select(start, start + s.length());
            } else {
                zoneTexte.replaceSelection(stab);
            }
        }
    }
    
    protected JMenuItem ajouterMenu(JMenu parent, String titre, String commande, int touche) {
        JMenuItem menu = new JMenuItem(titre);
        if (touche != 0) {
            int cmdMenu = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
            menu.setAccelerator(KeyStroke.getKeyStroke(touche, cmdMenu));
        }
        menu.addActionListener(this);
        menu.setActionCommand(commande);
        parent.add(menu).setEnabled(true);
        return(menu);
    }
    
    protected void initMenus() {
        JMenuBar barreMenus = new JMenuBar();
        
        JMenu menuFichier = new JMenu("Fichier");
        ajouterMenu(menuFichier, "Nouveau", "nouveau", java.awt.event.KeyEvent.VK_N);
        ajouterMenu(menuFichier, "Ouvrir...", "ouvrir", java.awt.event.KeyEvent.VK_O);
        menuFichier.addSeparator();
        ajouterMenu(menuFichier, "Fermer", "fermer", java.awt.event.KeyEvent.VK_W);
        ajouterMenu(menuFichier, "Enregistrer", "enregistrer", java.awt.event.KeyEvent.VK_S);
        ajouterMenu(menuFichier, "Enregistrer Sous...", "enregistrerSous", 0);
        menuFichier.addSeparator();
        ajouterMenu(menuFichier, "Imprimer...", "imprimer", java.awt.event.KeyEvent.VK_P);
        if (!System.getProperty("os.name").startsWith("Mac")) {
            menuFichier.addSeparator();
            ajouterMenu(menuFichier, "Quitter", "quitter", java.awt.event.KeyEvent.VK_Q);
        }
        barreMenus.add(menuFichier);
        
        JMenu menuEdition = new JMenu("Edition");
        menuAnnuler = ajouterMenu(menuEdition, "Annuler", "annuler", java.awt.event.KeyEvent.VK_Z);
        menuAnnuler.setEnabled(false);
        menuRetablir = ajouterMenu(menuEdition, "Rtablir", "retablir", java.awt.event.KeyEvent.VK_R);
        menuRetablir.setEnabled(false);
        menuEdition.addSeparator();
        ajouterMenu(menuEdition, "Couper", "couper", java.awt.event.KeyEvent.VK_X);
        ajouterMenu(menuEdition, "Copier", "copier", java.awt.event.KeyEvent.VK_C);
        ajouterMenu(menuEdition, "Coller", "coller", java.awt.event.KeyEvent.VK_V);
        menuEdition.addSeparator();
        ajouterMenu(menuEdition, "Rechercher", "rechercher", java.awt.event.KeyEvent.VK_F);
        barreMenus.add(menuEdition);
        
        setJMenuBar(barreMenus);
    }
    
    protected File dialogueOuvrir() {
        FileDialog fdlg = new FileDialog(this, "Choisir le fichier XML", FileDialog.LOAD);
        //fdlg.setFilenameFilter(new ExtFilter("xml"));
        fdlg.show();
        String chemin = null;
        String dir = fdlg.getDirectory();
        if (dir != null && dir.endsWith(File.separator))
            dir = dir.substring(0, dir.length()-1);
        String nom = fdlg.getFile();
        if (dir == null)
            chemin = nom;
        else if (nom != null)
            chemin = dir + File.separator + nom;
        if (chemin != null)
            return(new File(chemin));
        else
            return(null);
    }
    
    protected void detecterEncodage(String ligne) {
        if (ligne != null && ligne.startsWith("<?xml ")) {
            int ind = ligne.indexOf("encoding");
            if (ind == -1)
                return;
            ligne = ligne.substring(ind);
            ind = ligne.indexOf('"');
            int ind2 = ligne.indexOf('\'');
            char cs = '"';
            if (ind == -1 && ind2 != -1) {
                ind = ind2;
                cs = '\'';
            }
            if (ind == -1)
                return;
            ind2 = ligne.substring(ind+1).indexOf(cs);
            if (ind2 == -1)
                return;
            encodage = ligne.substring(ind+1, ind2+ind+1);
        }
    }
    
    protected void detecterEncodage(File f) {
        try {
            BufferedReader in = new BufferedReader(new FileReader(f));
            String ligne = in.readLine();
            if (ligne != null)
                detecterEncodage(ligne);
            in.close();
        } catch (IOException ex) {
            System.err.println("IOException: " + ex.getMessage());
        }
        
    }
    
    protected void detecterEncodage() {
        Element para1 = doc.getParagraphElement(0);
        int debut = para1.getStartOffset();
        int fin = para1.getEndOffset();
        try {
            String ligne = zoneTexte.getText(debut, fin-debut);
            detecterEncodage(ligne);
        } catch (BadLocationException ex) {
            System.err.println("BadLocationException: " + ex.getMessage());
        }
    }
    
    /**
     * Ouverture d'un document XML, dans une nouvelle fentre si ce document est vide.
     */
    public void ouvrir(File fichierXML) {
        if (fichierXML == null) {
            File f = dialogueOuvrir();
            if (f == null)
                return;
            if (f.equals(this.fichierXML))
                return;
            fichierXML = f;
            if (doc.getLength() > 0) {
                Mex.ouvrir(fichierXML);
                return;
            }
        }
        this.fichierXML = fichierXML;
        ignorerModifs = true;
        detecterEncodage(fichierXML);
        try {
            BufferedReader in = new BufferedReader(new InputStreamReader(
                new FileInputStream(fichierXML), encodage));
            zoneTexte.read(in, null);
        } catch (IOException ex) {
            ignorerModifs = false;
            System.err.println("IOException: " + ex.getMessage());
            return;
        }
        
        doc = zoneTexte.getStyledDocument();
        changementDoc();
        
        toutColorier();
        setTitle(fichierXML.getName());
        
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                ignorerModifs = false;
                modif = false;
            }
        });
        
        setVisible(true);
        zoneTexte.requestFocus();
    }
    
    class Buffer {
        int tailleMax = 200;
        String sbuff;
        int debut, fin; // fin n'inclut pas le dernier caractre du buffer
        
        public Buffer() {
            lire(0);
        }
        
        public void lire(int ind) {
            int lg = tailleMax;
            if (ind+lg > doc.getLength())
                lg = doc.getLength() - ind;
            try {
                sbuff = doc.getText(ind, lg);
            } catch (BadLocationException ex) {
                System.err.println("BadLocationException: " + ex.getMessage());
            }
            if (sbuff.length() != lg)
                System.err.println("erreur: " + sbuff.length() + " != " + lg);
            debut = ind;
            fin = ind + lg;
        }
        
        public char getChar(int p) {
            if (p >= doc.getLength())
                return(' ');
            if (p >= fin)
                lire(p);
            else if (p < debut) {
                int p2 = p - tailleMax + 1;
                if (p2 < 0)
                    p2 = 0;
                lire(p2);
            }
            return(sbuff.charAt(p-debut));
        }
        /*
        public String getString(int ind, int lg) {
            if (ind >= doc.getLength())
                System.err.println("erreur dans Buffer.getString: ind >= doc.getLength() : " + ind + " >= " + doc.getLength());
            if (ind+lg > doc.getLength())
                lg = doc.getLength() - ind;
            if (lg > tailleMax)
                System.err.println("erreur dans Buffer.getString: " + lg + " > taille maxi (" + tailleMax + ")");
            if (ind < debut || ind+lg > fin)
                lire(ind);
            return(sbuff.substring(ind-debut, ind-debut+lg));
        }
        */
        public boolean subEquals(String s, int ind) {
            if (ind >= doc.getLength())
                System.err.println("erreur dans Buffer.subEquals: ind >= doc.getLength() : " + ind + " >= " + doc.getLength());
            int lg = s.length();
            if (ind + lg >= doc.getLength())
                return(false);
            if (lg > tailleMax)
                System.err.println("erreur dans Buffer.subEquals: " + lg + " > taille maxi (" + tailleMax + ")");
            if (ind < debut || ind+lg > fin)
                lire(ind);
            for (int i=0, j=ind-debut; i<lg; i++,j++)
                if (s.charAt(i) != sbuff.charAt(j))
                    return(false);
            return(true);
        }
    }
    
    /**
     * Met  jour les couleurs dans l'intervalle indiqu
     */
    public void colorier(int debut, int fin) {
        boolean vignore = ignorerModifs;
        ignorerModifs = true;
        
        // pb: bug de la JVM 1.4.2 sur MacOS X: le curseur n'est pas bien positionn
        // quand on clique sur un caractre aprs un setCharacterAttributes
        // mauvaise mise  jour de modelToView ?
        // autre bug: si le premier caractre est un \n, les autres caractres disparaissent aprs undo !!!
        Buffer buff = new Buffer();
        if (buff.subEquals("\n", debut)) {
            if (fin - debut > 1)
                doc.setCharacterAttributes(debut+1, fin-debut-1, styleTexte, false);
        } else
            doc.setCharacterAttributes(debut, fin-debut, styleTexte, false);
        
        boolean dansNomElement = false;
        boolean dansNomAttribut = false;
        boolean avantValeurAttribut = false;
        char carValeurAttribut = '"';
        boolean dansValeurAttribut = false;
        boolean dansEntite = false;
        boolean dansCommentaire = false;
        int debutzone = debut;
        for (int ic=debut; ic<fin; ic++) {
            if (dansCommentaire) {
                if (buff.subEquals("-->", ic)) {
                    dansCommentaire = false;
                    ic += 2;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleCommentaire, false);
                }
            } else if (dansNomElement) {
                char c = buff.getChar(ic);
                if (c == ' ' || c == '\n') {
                    dansNomElement = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone, styleElement, false);
                    dansNomAttribut = true;
                    debutzone = ic+1;
                } else if (c == '>' || ic == fin-1) {
                    dansNomElement = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleElement, false);
                }
            } else if (dansNomAttribut) {
                char c = buff.getChar(ic);
                if (c == '>' || ic == fin-1) {
                    dansNomAttribut = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleElement, false);
                } else if (c == '=') {
                    dansNomAttribut = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone, styleNomAttribut, false);
                    avantValeurAttribut=true;
                }
            } else if (avantValeurAttribut) {
                char c = buff.getChar(ic);
                if (c == '"' || c=='\'') {
                    avantValeurAttribut = false;
                    dansValeurAttribut = true;
                    carValeurAttribut = c;
                    debutzone = ic;
                }
            } else if (dansValeurAttribut) {
                char c = buff.getChar(ic);
                if (c == carValeurAttribut || ic == fin-1) {
                    dansValeurAttribut = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleValeurAttribut, false);
                    dansNomAttribut = true;
                    debutzone = ic+1;
                }
            } else if (dansEntite) {
                char c = buff.getChar(ic);
                if (c == ';' || c == ' ' || c == '\n' || ic == fin-1) {
                    dansEntite = false;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleEntite, false);
                }
            } else {
                char c = buff.getChar(ic);
                if (c == '<') {
                    if (buff.subEquals("<!--", ic))
                        dansCommentaire = true;
                    else
                        dansNomElement = true;
                    debutzone = ic;
                } else if (c == '>') {
                    doc.setCharacterAttributes(ic, 1, styleElement, false);
                } else if (c == '&' || c == '%') {
                    dansEntite = true;
                    debutzone = ic;
                }
                
            }
        }
        
        // si on est toujours dans un commentaire  la fin de la zone, on continue  colorier au-del
        if (dansCommentaire) {
            for (int ic=fin; ic<doc.getLength() && dansCommentaire; ic++) {
                if (buff.subEquals("-->", ic)) {
                    dansCommentaire = false;
                    ic += 2;
                    doc.setCharacterAttributes(debutzone, ic-debutzone+1, styleCommentaire, false);
                }
            }
            if (dansCommentaire)
                doc.setCharacterAttributes(debutzone, doc.getLength()-debutzone, styleCommentaire, false);
        }
        ignorerModifs = vignore;
    }
    
    protected int debutMajCouleurs(int pos) {
        Buffer buff = new Buffer();
        boolean commentaire;
        Element elpc = doc.getCharacterElement(pos);
        commentaire = elpc.getAttributes().containsAttributes(styleCommentaire);
        if (!commentaire && pos > 2 && pos < doc.getLength()) {
            elpc = doc.getCharacterElement(pos-1);
            if (elpc.getAttributes().containsAttributes(styleCommentaire))
                if (!buff.subEquals("-->", pos - 3))
                    commentaire = true;
        }
        if (commentaire) {
            for (int ic=pos; ic>=0; ic--) {
                if (buff.subEquals("<!--", ic))
                    return(ic);
            }
        } else {
            for (int ic=pos; ic>=0; ic--) {
                char c = buff.getChar(ic);
                if (c == '<')
                    return(ic);
                else if (c == '>' && ic < pos)
                    return(ic+1);
            }
        }
        return(0);
    }
    
    protected int finMajCouleurs(int pos) {
        if (pos == 0)
            return(pos);
        Buffer buff = new Buffer();
        Element elpc = doc.getCharacterElement(pos-1);
        //pb ici: parfois le test ne marche pas aprs que l'on tape "-->"
        if (elpc.getAttributes().containsAttributes(styleCommentaire)) {
            for (int ic=pos; ic<doc.getLength(); ic++) {
                if (buff.subEquals("-->", ic))
                    return(ic+3);
            }
        } else {
            for (int ic=pos; ic<doc.getLength(); ic++) {
                char c = buff.getChar(ic);
                if (c == '>')
                    return(ic+1);
            }
        }
        return(doc.getLength());
    }
    
    protected void majCouleurs(int pos, int longueur) {
        poscol = debutMajCouleurs(pos);
        fincol = finMajCouleurs(pos + longueur);
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                colorier(poscol, fincol);
            }
        });
    }
    
    /**
     * Met  jour les couleurs dans tout le document
     */
    public void toutColorier() {
        colorier(0, doc.getLength()-1);
    }
    
    /**
     * Affiche un dialogue pour enregistrer le document avant une opration s'il a t modifi.
     * Renvoit true sans rien demander si le document n'a pas t modifi.
     * Si l'utilisateur rpond oui, enregistre le document et renvoit true.
     * Si l'utilisateur rpond non, renvoit true.
     * Si l'utilisateur rpond annuler, renvoit false.
     */
    public boolean testEnregistrerAvant() {
        if (modif) {
            int test = JOptionPane.showConfirmDialog(this, "Enregistrer avant ?", "",
                    JOptionPane.YES_NO_CANCEL_OPTION);
            if (test == JOptionPane.YES_OPTION)
                enregistrer(fichierXML);
            return(test != JOptionPane.CANCEL_OPTION);
        }
        return(true);
    }
    
    /**
     * Renvoit true si le document est vide
     */
    public boolean estVide() {
        return(doc != null && doc.getLength() == 0);
    }
    
    /**
     * Renvoit le fichier du document, ou null s'il n'a pas t spcifi
     */
    public File fichierCorrespondant() {
        return(fichierXML);
    }
    
    protected File dialogueEnregistrement() {
        FileDialog fd = new FileDialog(this, null, FileDialog.SAVE);
        fd.show();
        String sf = fd.getFile();
        if (sf == null)
            return(null);
        File f = new File(fd.getDirectory(), sf);
        if (f.getName().indexOf('.') == -1) {
            f = new File(f.getPath() + ".xml");
            if (f.exists()) {
                if (JOptionPane.showConfirmDialog(this, "Ce fichier existe dj. Le remplacer ?", "",
                    JOptionPane.YES_NO_OPTION) == JOptionPane.NO_OPTION) {
                        return(null);
                    }
            }
        }
        if (fvalidation != null) {
            fvalidation.setVisible(false);
            fvalidation = null;
        }
        return(f);
    }
    
    /**
     * Enregistre le document dans le fichier spcifi
     */
    public void enregistrer(File f) {
        if (f == null)
            f = dialogueEnregistrement();
        if (f == null)
            return;
        
        detecterEncodage();
        
        Writer out;
        try {
            out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(f), encodage));
        } catch (Exception ex) {
            JOptionPane.showMessageDialog(this, ex.getLocalizedMessage(), "Enregistrement",
                JOptionPane.ERROR_MESSAGE);
            return;
        }
        try {
            zoneTexte.write(out);
            out.close();
            modif = false;
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this, ex.getLocalizedMessage(), "Erreur d'entre/sortie",
                JOptionPane.ERROR_MESSAGE);
        }
        fichierXML = f;
        setTitle(fichierXML.getName());
    }
    
    /**
     * Enregistre le document en demandant au pralable o il doit tre enregistr
     */
    public void enregistrerSous() {
        File f = dialogueEnregistrement();
        if (f == null)
            return;
        enregistrer(f);
    }
    
    /**
     * Imprime le document
     */
    public void imprimer() {
        DocumentRenderer renderer = new DocumentRenderer();
        //renderer.pageDialog();
        renderer.print(zoneTexte);
    }
    
    // pour que l'annulation de l'entre "blabla" ne donne pas "blabl" mais ""
    protected class MyEdit extends AbstractUndoableEdit {
        public final static int ERREUR = 0;
        public final static int AJOUTER = 1;
        public final static int SUPPRIMER = 2;
        private int type;
        private int offset;
        private int length;
        private Vector edits;
        
        public MyEdit(UndoableEdit anEdit) {
            super();
            edits = new Vector();
            if (anEdit instanceof AbstractDocument.DefaultDocumentEvent) {
                AbstractDocument.DefaultDocumentEvent event = (AbstractDocument.DefaultDocumentEvent)anEdit;
                if (event.getType() == DocumentEvent.EventType.INSERT)
                    type = AJOUTER;
                else if (event.getType() == DocumentEvent.EventType.REMOVE)
                    type = SUPPRIMER;
                else
                    type = ERREUR;
                offset = event.getOffset();
                length = event.getLength();
            } else
                type = ERREUR;
            edits.add(anEdit);
        }
        public boolean addEdit(UndoableEdit anEdit) {
            MyEdit mEdit = (MyEdit)anEdit;
            if ((type == AJOUTER && mEdit.getOffset() == offset + length) ||
                    (type == SUPPRIMER && mEdit.getOffset() == offset)) {
                edits.add(mEdit.getEvent());
                length += mEdit.getLength();
                return true;
            }
            if ((type == AJOUTER && mEdit.getOffset() == offset) ||
                (type == SUPPRIMER && mEdit.getOffset() == offset - mEdit.getLength())) {
                edits.add(mEdit.getEvent());
                length += mEdit.getLength();
                offset -= mEdit.getLength();
                return true;
            }
            return(false);
        }
        public AbstractDocument.DefaultDocumentEvent getEvent() {
            if (edits.size() == 1)
                return (AbstractDocument.DefaultDocumentEvent)edits.get(0);
            else
                return null;
        }
        public int getType() {
            return type;
        }
        public int getOffset() {
            return offset;
        }
        public int getLength() {
            return length;
        }
        public void undo() throws CannotUndoException {
            super.undo();
            int i = edits.size();
            while (i-- > 0) {
                UndoableEdit e = (UndoableEdit)edits.elementAt(i);
                e.undo();
            }
        }
        public void redo() throws CannotRedoException {
            super.redo();
            Enumeration cursor = edits.elements();
            while (cursor.hasMoreElements())
                ((UndoableEdit)cursor.nextElement()).redo();
        }
        public boolean canUndo() {
            int count = edits.size();
            if (count > 0)
                return ((UndoableEdit)edits.elementAt(count-1)).canUndo();
            else
                return false;
        }
        public boolean canRedo() {
            int count = edits.size();
            if (count > 0)
                return ((UndoableEdit)edits.elementAt(0)).canRedo();
            else
                return false;
        }
    }
    
    protected class MonEcouteurDAnnulations implements UndoableEditListener {
        public void undoableEditHappened(UndoableEditEvent e) {
            if (!ignorerModifs) {
                undoMgr.addEdit(new MyEdit(e.getEdit()));
                majMenusAnnulation();
            }
        }
    }
    
    protected void majMenusAnnulation() {
        menuAnnuler.setEnabled(undoMgr.canUndo());
        menuRetablir.setEnabled(undoMgr.canRedo());
    }
    
    /**
     * Annule la dernire opration (affiche un message d'erreur si ce n'est pas possible)
     */
    public void annuler() {
        try {
            undoMgr.undo();
        } catch (CannotUndoException ex) {
            JOptionPane.showMessageDialog(this, "Impossible d'annuler: " + ex.getLocalizedMessage(),
                "Annulation", JOptionPane.ERROR_MESSAGE);
        }
        majMenusAnnulation();
    }
    
    /**
     * Rtablit la dernire opration (affiche un message d'erreur si ce n'est pas possible)
     */
    public void retablir() {
        try {
            undoMgr.redo();
        } catch (CannotRedoException ex) {
            JOptionPane.showMessageDialog(this, "Impossible de rtablir: " + ex.getLocalizedMessage(),
                "Rtablir", JOptionPane.ERROR_MESSAGE);
        }
        majMenusAnnulation();
    }
    
    // affiche le dialogue de recherche
    public void rechercher() {
        DialogueRechercher dlg = new DialogueRechercher(zoneTexte);
        dlg.show();
    }
    
    public void actionPerformed(ActionEvent e) {
        String cmd = e.getActionCommand();
        
        if ("valider".equals(cmd))
            valider();
        
        else if ("nouveau".equals(cmd))
            Mex.nouveau();
        else if ("ouvrir".equals(cmd))
            ouvrir(null);
        else if ("fermer".equals(cmd))
            Mex.fermer(this);
        else if ("enregistrer".equals(cmd))
            enregistrer(fichierXML);
        else if ("enregistrerSous".equals(cmd))
            enregistrerSous();
        else if ("imprimer".equals(cmd))
            imprimer();
        else if ("quitter".equals(cmd))
            Mex.quitter(true);
        
        else if ("annuler".equals(cmd))
            annuler();
        else if ("retablir".equals(cmd))
            retablir();
        else if ("couper".equals(cmd))
            zoneTexte.cut();
        else if ("copier".equals(cmd))
            zoneTexte.copy();
        else if ("coller".equals(cmd))
            zoneTexte.paste();
        else if ("rechercher".equals(cmd))
            rechercher();
    }
    
    /**
     * Positionne le document  la ligne indique (la premire ligne a le numro 1)
     */
    public void allerLigne(int ligne) {
        if (ligne > 0)
            ligne--;
        else
            ligne = 0;
        int pos = doc.getDefaultRootElement().getElement(ligne).getStartOffset();
        // bidouille pour afficher la position en haut de la fentre
        try {
            zoneTexte.scrollRectToVisible(zoneTexte.modelToView(doc.getLength()));
            zoneTexte.scrollRectToVisible(zoneTexte.modelToView(pos));
        } catch (BadLocationException ex) {
            System.err.println(ex.getClass().getName() + " " + ex.getLocalizedMessage());
        }
    }
    
    /**
     * Slectionne la ligne indique (la premire ligne a le numro 1)
     */
    public void selectLigne(int ligne) {
        if (ligne > 0)
            ligne--;
        else
            ligne = 0;
        Element ligneel = doc.getDefaultRootElement().getElement(ligne);
        try {
            zoneTexte.scrollRectToVisible(zoneTexte.modelToView(ligneel.getStartOffset()));
        } catch (BadLocationException ex) {
            System.err.println(ex.getClass().getName() + " " + ex.getLocalizedMessage());
        }
        zoneTexte.setCaretPosition(ligneel.getStartOffset());
        if (ligneel.getEndOffset() <= doc.getLength())
            zoneTexte.moveCaretPosition(ligneel.getEndOffset());
        else
            zoneTexte.moveCaretPosition(doc.getLength());
        zoneTexte.requestFocus();
    }
    
    /**
     * Lance la validation du document, avec l'affichage ou la mise  jour de la fentre de validation
     */
    public void valider() {
        if (!testEnregistrerAvant())
            return;
        if (fvalidation == null)
            fvalidation = new FenetreValidation(fichierXML, this);
        else
            fvalidation.miseAJour();
    }
    
    /**
     * Spcifie la taille des tabulations, en quivalent-caractres (on utilise la taille du 'w' comme rfrence)
     */
    public void setTabs(int charactersPerTab) {
        FontMetrics fm = zoneTexte.getFontMetrics(zoneTexte.getFont());
        int charWidth = fm.charWidth('w');
        int tabWidth = charWidth * charactersPerTab;
        
        TabStop[] tabs = new TabStop[10];
        
        for (int j = 0; j < tabs.length; j++) {
            int tab = j + 1;
            tabs[j] = new TabStop( tab * tabWidth );
        }
        
        TabSet tabSet = new TabSet(tabs);
        SimpleAttributeSet attributes = new SimpleAttributeSet();
        StyleConstants.setTabSet(attributes, tabSet);
        int length = doc.getLength();
        doc.setParagraphAttributes(0, length, attributes, false);
    }
    
}
