/*
    Copyright (C) 2004  Damien Guillaume
    
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* A frame to display a clustering grid
	initial version for AML documents, Damien Guillaume, 2/12/98
	new version for random documents, Damien Guillaume, 15/4/2004
*/

package dispgrid;

import java.applet.Applet;
import java.io.*;
import java.net.*;
import java.util.Vector;
import java.util.Random;
import java.awt.*;
import java.awt.event.*;

import xml.*;


/**
 * A frame to display a clustering grid.<br>
 * This class is using a "GRID" XMLTree as input,
 * describing the clusters and the nodes they control
 *
 * @version     2.0, 15/4/2004
 * @author      Damien Guillaume
 * @see         xml.Parser_XML
 * @see         xml.XMLTree
 * @see         clustering.Clustering
 * @see         DispGrid
 */

public class GridFrame extends Frame implements ActionListener, ItemListener {

  boolean inited = false;
  FramedArea framedArea;
  List lsel;
  TextField gridurl_field;
  XMLTree gridTree=null;
  XMLTree bigFather=null;
  XMLTree currentCluster=null;
  Button bopendoc;
  Button bzoomp;
  Button bzoomm;
  TextField find_field;
  XMLTree selCluster=null; // selected Cluster on the grid
  XMLTree oldselag=null;
  Vector selnodes=null;
  ErrorDialog errdlg=null;
  Applet myappl=null;
  final static String DEFAULT_GRID = "http://server_to_define/folder/grid";

  public GridFrame(Applet appl) {
    super("Clustering");
    myappl = appl;
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
	e.getWindow().setVisible(false);
	e.getWindow().dispose();
      }
    });
  }
  
  private void error(String s) {
    if (inited) {
      System.err.println(s);
      if (errdlg == null)
        errdlg = new ErrorDialog(this, s);
      else
        errdlg.newMessage(s);
    } else
      System.err.println(s);
  }
  
  public void init() {
    GridBagLayout gridBag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();

    setLayout(gridBag);
    setFont(new Font("Helvetica", Font.PLAIN, 14));
    Panel cp = new Panel();
    GridBagLayout cpgridBag = new GridBagLayout();
    GridBagConstraints cpc = new GridBagConstraints();
    cp.setLayout(cpgridBag);
    Label lab = new Label("Grid URL:", Label.CENTER);
    cpgridBag.setConstraints(lab, cpc);
    cp.add(lab);
    gridurl_field = new TextField(DEFAULT_GRID, 40);
    gridurl_field.addActionListener(this);
    cpc.gridwidth = 1;
    cpgridBag.setConstraints(gridurl_field, cpc);
    cp.add(gridurl_field);

    Button bok = new Button("Open");
    cpgridBag.setConstraints(bok, cpc);
    cp.add(bok);
    bok.setActionCommand("OK");
    bok.addActionListener(this);
    c.fill = GridBagConstraints.HORIZONTAL;
    c.weightx = 0.0;
    c.weighty = 0.0;
    c.gridwidth = GridBagConstraints.REMAINDER; //end row
    gridBag.setConstraints(cp, c);
    add(cp);

    framedArea = new FramedArea(this);
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 1.0;
    c.weighty = 1.0;
    c.gridwidth = GridBagConstraints.REMAINDER; //end row
    gridBag.setConstraints(framedArea, c);
    add(framedArea);

    lsel = new List(6);
    c.weightx = 0.0;
    c.weighty = 0.0;
    //c.fill = GridBagConstraints.HORIZONTAL;
    c.fill = GridBagConstraints.BOTH;
    gridBag.setConstraints(lsel, c);
    lsel.addItemListener(this);
    lsel.addActionListener(this);
    add(lsel);

    Panel cpz = new Panel();
    
    bopendoc = new Button("Open selection");
    bopendoc.setEnabled(false);
    cpz.add(bopendoc);
    bopendoc.setActionCommand("opendoc");
    bopendoc.addActionListener(this);

    bzoomp = new Button("Zoom+");
    bzoomp.setEnabled(false);
    cpz.add(bzoomp);
    bzoomp.setActionCommand("zoomp");
    bzoomp.addActionListener(this);
    bzoomm = new Button("Zoom-");
    bzoomm.setEnabled(false);
    cpz.add(bzoomm);
    bzoomm.setActionCommand("zoomm");
    bzoomm.addActionListener(this);
    
    lab = new Label("Find:", Label.CENTER);
    cpz.add(lab);
    find_field = new TextField("", 20);
    find_field.addActionListener(this);
    cpz.add(find_field);
    
    gridBag.setConstraints(cpz, c);
    add(cpz);

    pack();
    check_myframe_position();
    show();
    
    inited = true;
  }

  public void check_myframe_position() {
    // to prevent windows outside the screen
    Insets is = getInsets();
    Point pt = getLocation();
    if (pt.x < is.left)
      pt.x = is.left;
    if (pt.y < is.top)
      pt.y = is.top; // the result is not correct, why ???
    Toolkit tool = getToolkit();
    Dimension dds = tool.getScreenSize();
    Dimension ddf = getSize();
    if (pt.x + ddf.width > dds.width)
      pt.x = dds.width - ddf.width;
    if (pt.y + ddf.height > dds.height)
      pt.y = dds.height - ddf.height + is.top;
    setLocation(pt.x,pt.y);
  }

  public XMLTree getAgNodeAt(int x, int y) {
    XMLTree node, nodefound;
    Attribute attx,atty;
    boolean trouv;
    String sx, sy;

    if (gridTree == null)
      return(null);
    sx = Integer.toString(x);
    sy = Integer.toString(y);
    //System.out.println("sx="+sx+", sy="+sy);
    trouv = false;
    nodefound = null;
    for (node=gridTree.first_child; !trouv && (node!= null); node=node.next_brother) {
      if (node.attr != null) {
	attx = (Attribute)node.attr.elementAt(0);
	atty = (Attribute)node.attr.elementAt(1);
	//System.out.println("attx="+attx.val+", atty="+atty.val);
	if (attx.val.equals(sx) && atty.val.equals(sy)) {
	  trouv = true;
	  nodefound = node;
	}
      }
    }
    /*if (nodefound != null)
      System.out.println("Found cluster "+nodefound.getAttVal("centerURL")+" at ("+x+","+y+")");
    else
      System.out.println("No cluster found at ("+x+","+y+")");*/
    return(nodefound);
  }

  public String decodeURL(String s) {
    int i;
    String res;
    Integer c;

    for (i=0,res=""; i<s.length(); i++)
      if (s.charAt(i)=='+')
	res += ' ';
      else if (s.charAt(i)=='%') {
	c = Integer.valueOf(s.substring(i+1,i+3),16);
	res += (char)(c.intValue());
	i += 2;
      } else
	res += s.charAt(i);
    //System.out.println(s + " -> " + res);
    return(res);
  }

  public String oneName(String sURL) {
    int i,j,k;

    if (sURL == null)
      return("");
    i = sURL.lastIndexOf('/'); // from a URL, keep only the file name
    if (i == -1)
      return(decodeURL(sURL));
    k = sURL.indexOf('='); // if it's a CGI, keep only the first arg
    if (k != -1) {
      i = k;
      j = sURL.indexOf('&');
      if (j == -1)
	j = sURL.length();
    } else
      j = sURL.lastIndexOf('.'); // remove file name extension
    if (i<j)
      return(decodeURL(sURL.substring(i+1,j)));
    else
      return(decodeURL(sURL.substring(i+1)));
  }

  /*  public String oneName(String sURL) {
    int i,j,k;

    //decodeURL(sURL);
    i = sURL.lastIndexOf('/'); // from a URL, keep only the file name
    if (i == -1)
      return(decodeURL(sURL));
    k = sURL.lastIndexOf('='); // if it's a CGI, keep only the last arg
    if (k > i) {
      i = k;
      j = sURL.length();
    } else
      j = sURL.lastIndexOf('.'); // remove file name extension
    if (i<j)
      return(decodeURL(sURL.substring(i+1,j)));
    else
      return(decodeURL(sURL.substring(i+1)));
  }*/

  public void addNodesToList(XMLTree ag) {
    XMLTree n;
    String orig,sinfo,title;

    for (n = ag.first_child; n != null; n = n.next_brother)
      if ((n.tag.equals("SLAVE"))||(n.tag.equals("NODE"))) {
      // "SLAVE" = "NODE", the tags have changed
	if (n.attr != null) {
	  title = n.getAttVal("title");
	  if ((orig = n.getAttVal("orig")) != null) {
	    if (orig.equals("1"))
	      sinfo = " ***";
	    else if (orig.equals("2"))
	      sinfo = " **";
	    else if (orig.equals("3"))
	      sinfo = " *";
	    else
	      sinfo = "";
	  } else
	    sinfo = "";
	  
	  if ((title != null) && (!title.equals("")))
	    lsel.add(title + sinfo);
	  else
	    lsel.add(oneName(n.getAttVal("url")) + sinfo);
	  selnodes.addElement(n);
	}
      } else if (n.tag.equals("CLUSTER"))
	addNodesToList(n);
  }

  public void modList(int x, int y) {
    XMLTree ag;

    if (lsel.getItemCount() > 0)
      lsel.removeAll();
    selnodes = new Vector();
    if ((ag=getAgNodeAt(x,y)) != null) {
      selCluster = ag;
      if ((ag.first_child == null) || !ag.first_child.tag.equals("CLUSTER"))
	bzoomp.setEnabled(false);
      else
	bzoomp.setEnabled(true);
      if ((ag.father == null) || !ag.father.tag.equals("CLUSTER"))
	bzoomm.setEnabled(false);
      else
	bzoomm.setEnabled(true);
      addNodesToList(ag);
    }
    currentCluster = ag;
    bopendoc.setEnabled(false);
    lsel.repaint();
    framedArea.repaint();
  }

  public void changeGridClusters(XMLTree tree) {
    int oldx, oldy;
    
    gridTree = tree;
    for (bigFather=gridTree; bigFather.father != null; bigFather=bigFather.father)
      ;
    framedArea.image = null;
    framedArea.repaint();
    if ((oldselag == null)||(oldselag.father != tree)) {
      oldselag = selCluster;
      selCluster = getAgNodeAt(0,0);
      modList(0,0);
    } else {
      oldx = Integer.decode(oldselag.getAttVal("x")).intValue();
      oldy = Integer.decode(oldselag.getAttVal("y")).intValue();
      oldselag = selCluster;
      modList(oldx, oldy);
    }
  }

  public void letsgo(String gridurl) {
    URL theURL;
    Parser_XML p;
    XMLTree thisTree=null;

    if (!gridurl.equals("")) {
      setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      gridurl_field.setText(gridurl);
      thisTree = new XMLTree(null,"",null,"");
      System.out.println("Reading GRID file: " + gridurl);
      if (gridurl.startsWith("http://")) {
        try {
            theURL = new URL(gridurl);
        } catch(Exception e) {
            error("Error in creating the URL connection");
            setCursor(Cursor.getDefaultCursor());
            return;
        }
        p = new Parser_XML(theURL, System.out, thisTree);
      } else {
        if (gridurl.startsWith("file://"))
            gridurl = gridurl.substring(7);
        try {
            p = new Parser_XML(new FileInputStream(gridurl), System.out, thisTree);
        } catch(Exception e) {
            error("Error in reading the grid file " + gridurl);
            setCursor(Cursor.getDefaultCursor());
            return;
        }
      }
      p.parse(0);
      
      if ((thisTree == null) || (!thisTree.tag.equals("GRID"))) {
	error("error reading GRID file");
	setCursor(Cursor.getDefaultCursor());
	return;
      }
      changeGridClusters(thisTree);
      setCursor(Cursor.getDefaultCursor());
    }
  }

  public XMLTree find_in(String s, XMLTree tree, int[] levels) {
  // levels[0] = current level, levels[1] = best level
    XMLTree n, n_found, best_n = null;
    String s_up, title_up;
    int best_level=100;
    int[] levels2;
    
    s_up = s.toUpperCase();
    for (n=tree.first_child; n!=null; n=n.next_brother) {
      title_up = (n.getAttVal("title")).toUpperCase();
      if (title_up.indexOf(s_up) != -1) {
        levels[1] = levels[0];
        return(n);
      }
    }
    
    for (n=tree.first_child; n!=null; n=n.next_brother) {
      levels2 = new int[2];
      levels2[0] = levels[0]+1;
      if ((n_found = find_in(s, n, levels2)) != null)
        if (levels2[1] < best_level) {
          best_level = levels2[1];
          best_n = n_found;
        }
      
    }
    if (best_n != null) {
      levels[1] = best_level;
      return(best_n);
    }

    return(null);
  }
  
  public void find(String s) {
    XMLTree n_found;
    int[] levels;
    
    if (bigFather == null)
      return;
    setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    levels = new int[2];
    levels[0] = 0;
    n_found = find_in(s, bigFather, levels);
    if (n_found != null) {
      if ((n_found.first_child != null) && (n_found.first_child.first_child != null))
        changeGridClusters(n_found);
      else if ((n_found.father != null) && (n_found.first_child != null)) {
        changeGridClusters(n_found.father);
        if (n_found.father.father != null)
          oldselag = n_found.father;
        modList(Integer.decode(n_found.getAttVal("x")).intValue(),
          Integer.decode(n_found.getAttVal("y")).intValue());
      } else if ((n_found.father != null) && (n_found.father.father != null)) {
        changeGridClusters(n_found.father.father);
        if (n_found.father.father.father != null)
          oldselag = n_found.father.father;
        modList(Integer.decode(n_found.father.getAttVal("x")).intValue(),
          Integer.decode(n_found.father.getAttVal("y")).intValue());
      }
    }
    setCursor(Cursor.getDefaultCursor());
  }
  
  public void actionPerformed(ActionEvent event) {
    String command = event.getActionCommand();

    //System.out.println("command: "+command);
    if (event.getSource() == (Object)gridurl_field)
      command = "OK";
    else if (event.getSource() == (Object)find_field)
      command = "find";
    else if (event.getSource() == (Object)lsel)
      command = "opendoc";
    if (command.equals("OK")) {
      letsgo(gridurl_field.getText());
    } else if (command.equals("zoomp") && (selCluster != null)) {
      if (selCluster.first_child != null)
	changeGridClusters(selCluster);
    } else if (command.equals("zoomm") && (selCluster != null)) {
      if (selCluster.father != null)
	changeGridClusters(selCluster.father.father);
    } else if (command.equals("opendoc")) {
      int i = lsel.getSelectedIndex();
      if ((i != -1) && (currentCluster != null)) {
	XMLTree n;
	n = (XMLTree)(selnodes.elementAt(i));
	if (n != null) {
	  String surl = n.getAttVal("url");
	  if (myappl != null && surl != null) {
            try {
                URL durl = new URL(surl);
                myappl.getAppletContext().showDocument(durl, "docwindow");
            } catch (MalformedURLException ex) {
                error("MalformedURLException: " + ex.getMessage());
            }
          }
	}
      }
    } else if (command.equals("find"))
      find(find_field.getText());
  }

  public void itemStateChanged(ItemEvent e) {
    int i;

    if (e.getStateChange() == ItemEvent.SELECTED) {
      bopendoc.setEnabled(true);
    }
  }

  public static void main(String[] args) {
    String surl;
    GridFrame gf;
    
    gf = new GridFrame(null);
    gf.init();
    if (args.length > 0) {
        surl = args[0];
        if (surl != null)
            gf.letsgo(surl);
    }
  }
}
