/*
    Copyright (C) 2004  Damien Guillaume
    
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package clustering;

import java.util.Vector;
import java.util.Random;

/**
 * A class to partition a graph with a noising partitioning method.<br>
 * Algorithm reference:<br>
 * Guillaume, Damien and Murtagh, Fionn (2000).
 * Clustering of XML documents, Computer Physics Communications, 127, 215-227
 *
 * @version     2.0, 15 April 2004
 * @author      Damien Guillaume
 * @see         clustering.Clustering
 * @see         clustering.Node
 * @see         clustering.Cluster
 */

public class NoisingPartitioning extends Thread {

  final int MAX_ZOOM = 3; // grid depth with subClusters
  final int DIMX = Clustering.DIMX;
  final int DIMY = Clustering.DIMY;
  final int MAX_NB_AGENTS = DIMX*DIMY;
  boolean use_keywords = false; // keyword nodes are fake nodes (weight=0)
  boolean use_exchanges = false;
  Clustering clusframe;
  int init_method;
  Vector init_nodes;
  Cluster[] init_clusters;
  float init_alpha;

  public NoisingPartitioning(Clustering clusframe, int init_method, Vector init_nodes,
        Cluster[] init_clusters, float init_alpha, boolean use_keywords) {
      this.clusframe = clusframe;
      this.init_method = init_method;
      this.init_nodes = init_nodes;
      this.init_clusters = init_clusters;
      this.init_alpha = init_alpha;
      this.use_keywords = use_keywords;
  }
  
  public void run() {
    if (init_method == 2)
        noisingPartitioning(init_nodes, init_clusters, init_alpha);
    else
        recursPartitioning(init_nodes, init_clusters, null, 1, init_alpha);
    clusframe.endClustering();
  }
  
  Random rand = new Random();

  private int randomIntBetween(int a, int b) { // included a and b

    return((int)(a+((long)rand.nextInt()-Integer.MIN_VALUE)/(((long)Integer.MAX_VALUE-Integer.MIN_VALUE)/(b-a+1))));
  }

  private float randomFloatBetween(float a, float b) { // included a and b

    return((float)(a+((long)rand.nextInt()-Integer.MIN_VALUE)/(((long)Integer.MAX_VALUE-Integer.MIN_VALUE)/(b-a+1))));
  }

  private static float balance_cost(int nb, float moyna) {
    return((nb - moyna)*(nb - moyna)/moyna);
  }

  private float calc_moyna(Vector allNodes, int nb_clusters) {
    float moyna;
    int i;
    Node iNode;
    
    for (moyna=0, i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      if (!iNode.fake)
        moyna++;
    }
    return(moyna/nb_clusters);
  }
  
  private void descent(Vector allNodes, Cluster[] clusters, int nb_clusters, float alpha) {
    int i,j,k,l,m;
    Node jNode, lNode, iNode, mNode;
    Cluster ag;
    float moyna;
    float dimbalance, dcutlinks, dimbalance1;
    float linkweight;
    int best_k, best_m;
    float best_score, temp_score;
    boolean change;
    int nb_loops_for_bug=0;
    float change_best_score=0;
    int nbr;
    int dx1, dy1, dx2, dy2;
    float[] tdimbalance2 = new float[nb_clusters]; // for an optimisation
    double[] tdcutlinks2 = new double[100]; // idem (100 is not a limit)
    if (use_keywords)
      moyna = calc_moyna(allNodes, nb_clusters);
    else
      moyna = ((float)allNodes.size()) / nb_clusters;

    //System.out.println("starting descent...");
    
    for (i=0; i<nb_clusters; i++) {
      if (use_keywords)
        tdimbalance2[i] = balance_cost(clusters[i].nbreal+1,moyna) - balance_cost(clusters[i].nbreal,moyna);
      else
        tdimbalance2[i] = balance_cost(clusters[i].nodes.size()+1,moyna) - balance_cost(clusters[i].nodes.size(),moyna);
    }
    
    change = true;
    while (change) {
      //System.out.println("New loop");
      change = false;

      for (i=0; i<allNodes.size(); i++)
        ((Node)allNodes.elementAt(i)).mark = false;

      for (i=0; i<nb_clusters; i++) {
	ag = clusters[i];
        if (use_keywords)
	  dimbalance1 = balance_cost(ag.nbreal-1,moyna) - balance_cost(ag.nbreal,moyna);
        else
	  dimbalance1 = balance_cost(ag.nodes.size()-1,moyna) - balance_cost(ag.nodes.size(),moyna);

	for (j=0; j<ag.nodes.size(); j++) {
	  jNode = (Node)ag.nodes.elementAt(j);
	  best_k = -1;
	  best_score = 1000;
          
          if (jNode.links.size() > tdcutlinks2.length)
            tdcutlinks2 = new double[jNode.links.size()];
	  for (l=0; l<jNode.links.size(); l++) {
	    lNode = (Node)jNode.links.elementAt(l);
            dx2 = ag.x - lNode.master.x;
            dy2 = ag.y - lNode.master.y;
	    tdcutlinks2[l] = Math.sqrt(dx2*dx2 + dy2*dy2);
	  }
          
	  for (k=0; k<nb_clusters; k++)
	    if (k != i) {
	      // we study the move of jNode from clusters[i] to clusters[k]
              
             if (jNode.fake)
               dimbalance = 0;
             else
               dimbalance = dimbalance1;
              
              if (!jNode.fake)
                dimbalance += tdimbalance2[k];
              /*if (use_keywords) {
                if (!jNode.fake)
	          dimbalance += balance_cost(clusters[k].nbreal+1,moyna) - balance_cost(clusters[k].nbreal,moyna);
              } else
	        dimbalance += balance_cost(clusters[k].nodes.size()+1,moyna) - balance_cost(clusters[k].nodes.size(),moyna);
              */
              
	      dcutlinks = 0;
	      for (l=0; l<jNode.links.size(); l++) {
		lNode = (Node)jNode.links.elementAt(l);
		linkweight = ((MyFloat)jNode.roles.elementAt(l)).val;
                dx1 = lNode.master.x - clusters[k].x;
                dy1 = lNode.master.y - clusters[k].y;
                /*dx2 = ag.x - lNode.master.x;
                dy2 = ag.y - lNode.master.y;
		dcutlinks += linkweight * (Math.sqrt(dx1*dx1 + dy1*dy1) - Math.sqrt(dx2*dx2 + dy2*dy2));
                */
                dcutlinks += linkweight * (Math.sqrt(dx1*dx1 + dy1*dy1) - tdcutlinks2[l]);
	      }
	      if ((temp_score = dimbalance + alpha*dcutlinks) < best_score) {
		best_score = temp_score;
		best_k = k;
	      }
	    } // for k
	  if (best_score < -0.00001) {
	    // do the move !
	    jNode.master = clusters[best_k];
	    clusters[best_k].nodes.addElement(jNode);
	    ag.nodes.removeElementAt(j);
            if (!jNode.fake) {
              clusters[best_k].nbreal++;
              ag.nbreal--;
              if (use_keywords)
	        dimbalance1 = balance_cost(ag.nbreal-1,moyna) - balance_cost(ag.nbreal,moyna);
              else
	        dimbalance1 = balance_cost(ag.nodes.size()-1,moyna) - balance_cost(ag.nodes.size(),moyna);
              if (use_keywords)
                tdimbalance2[i] = balance_cost(clusters[i].nbreal+1,moyna) - balance_cost(clusters[i].nbreal,moyna);
              else
                tdimbalance2[i] = balance_cost(clusters[i].nodes.size()+1,moyna) - balance_cost(clusters[i].nodes.size(),moyna);
              if (use_keywords)
                tdimbalance2[best_k] = balance_cost(clusters[best_k].nbreal+1,moyna) - balance_cost(clusters[best_k].nbreal,moyna);
              else
                tdimbalance2[best_k] = balance_cost(clusters[best_k].nodes.size()+1,moyna) - balance_cost(clusters[best_k].nodes.size(),moyna);
            }
            
            // look at the linked nodes that could possibly move too
            // but that have already been tested in the i loop
	    for (l=0; l<jNode.links.size(); l++) {
	      lNode = (Node)jNode.links.elementAt(l);
	      linkweight = ((MyFloat)jNode.roles.elementAt(l)).val;
              if ((lNode.master.y < ag.y) || ((lNode.master.y == ag.y) && (lNode.master.x < ag.x)))
                if (jNode.master != lNode.master) {
                  if (linkweight > (lNode.links.size()-1)/20.0)
                    lNode.mark = true;
                }
            }
            
            /*
	    if (ag.centerURL.equals(jNode.url)) {
	      if (ag.nodes.size()>0) {
		ag.centerURL = ((Node)ag.nodes.elementAt(0)).url;
		ag.title = ((Node)ag.nodes.elementAt(0)).title;
		if (ag.title.equals(""))
		  ag.title = ((Node)ag.nodes.elementAt(0)).ident;
	      } else {
		ag.centerURL = "";
		ag.title = "";
	      }
	    }
	    if (clusters[best_k].centerURL.equals("")) {
	      clusters[best_k].centerURL = jNode.url;
	      if (jNode.title.equals(""))
		clusters[best_k].title = jNode.ident;
	      else
		clusters[best_k].title = jNode.title;
	    }
            */
	    j--;
	    change = true;
	    //System.out.println("Moved "+jNode.url+" from agent "+i+" to agent "+best_k+"(score="+best_score+")");
	    change_best_score = best_score;
	  } else if (use_exchanges && (best_k != -1)) {// if (best_score < -0.00001)
            // test exchanges between jNode and a node of agent best_k
            k = best_k;
	    best_score = 0;
            best_m = -1;
	    for (m=0; m<clusters[k].nodes.size(); m++) {
              mNode = (Node)clusters[k].nodes.elementAt(m);
              if (mNode.fake == jNode.fake) { // no dimbalance
	        // we study the exchange between jNode of clusters[i] and mNode of clusters[k]

	        dcutlinks = 0;
	        for (l=0; l<jNode.links.size(); l++) {
		  lNode = (Node)jNode.links.elementAt(l);
                  if (lNode != mNode) {
		    linkweight = ((MyFloat)jNode.roles.elementAt(l)).val;
                    dx1 = lNode.master.x - clusters[k].x;
                    dy1 = lNode.master.y - clusters[k].y;
                    dcutlinks += linkweight * (Math.sqrt(dx1*dx1 + dy1*dy1) - tdcutlinks2[l]);
                  }
	        }
	        for (l=0; l<mNode.links.size(); l++) {
		  lNode = (Node)mNode.links.elementAt(l);
                  if (lNode != jNode) {
		    linkweight = ((MyFloat)mNode.roles.elementAt(l)).val;
                    dx1 = lNode.master.x - ag.x;
                    dy1 = lNode.master.y - ag.y;
                    dx2 = clusters[k].x - lNode.master.x;
                    dy2 = clusters[k].y - lNode.master.y;
		    dcutlinks += linkweight * (Math.sqrt(dx1*dx1 + dy1*dy1) - Math.sqrt(dx2*dx2 + dy2*dy2));
                  }
	        }
	        if ((temp_score = dcutlinks) < best_score) {
		  best_score = temp_score;
                  best_m = m;
	        }
              }
            } // for m
	    if (best_score < -0.00001) {
	      // do the exchange !
              mNode = (Node)clusters[best_k].nodes.elementAt(best_m);
              
	      ag.nodes.removeElementAt(j);
	      clusters[best_k].nodes.removeElementAt(best_m);

	      jNode.master = clusters[best_k];
	      clusters[best_k].nodes.insertElementAt(jNode, best_m);
              
	      mNode.master = ag;
	      ag.nodes.insertElementAt(mNode,j);

              // look at the linked nodes that could possibly move too
              // but that have already been tested in the i loop
	      for (l=0; l<jNode.links.size(); l++) {
	        lNode = (Node)jNode.links.elementAt(l);
	        linkweight = ((MyFloat)jNode.roles.elementAt(l)).val;
                if ((lNode.master.y < ag.y) || ((lNode.master.y == ag.y) && (lNode.master.x < ag.x)))
                  if (jNode.master != lNode.master) {
                    if (linkweight > (lNode.links.size()-1)/20.0)
                      lNode.mark = true;
                  }
              }

	      for (l=0; l<mNode.links.size(); l++) {
	        lNode = (Node)mNode.links.elementAt(l);
	        linkweight = ((MyFloat)mNode.roles.elementAt(l)).val;
                if ((lNode.master.y < ag.y) || ((lNode.master.y == ag.y) && (lNode.master.x < ag.x)))
                  if (mNode.master != lNode.master) {
                    if (linkweight > (lNode.links.size()-1)/20.0)
                      lNode.mark = true;
                  }
              }

	      change = true;
	      //System.out.println("Exchanged "+jNode.url+" of agent "+i+" with node "+mNode.url+" of agent "+best_k+"(score="+best_score+")");
	      change_best_score = best_score;
	    } // if (best_score < -0.00001)

          } // if (best_score < -0.00001)...else if
	} // for j
      } // for i
            
      // test for nodes which could possibly move because
      // of their neighbours' move
      if (change)
      for (i=0; i<nb_clusters; i++) {
	ag = clusters[i];
        if (use_keywords)
	  dimbalance1 = balance_cost(ag.nbreal-1,moyna) - balance_cost(ag.nbreal,moyna);
        else
	  dimbalance1 = balance_cost(ag.nodes.size()-1,moyna) - balance_cost(ag.nodes.size(),moyna);

	for (j=0; j<ag.nodes.size(); j++)
         if (((Node)ag.nodes.elementAt(j)).mark) { // here is the change !
	  jNode = (Node)ag.nodes.elementAt(j);
	  best_k = -1;
	  best_score = 0;
          
          if (jNode.links.size() > tdcutlinks2.length)
            tdcutlinks2 = new double[jNode.links.size()];
	  for (l=0; l<jNode.links.size(); l++) {
	    lNode = (Node)jNode.links.elementAt(l);
            dx2 = ag.x - lNode.master.x;
            dy2 = ag.y - lNode.master.y;
	    tdcutlinks2[l] = Math.sqrt(dx2*dx2 + dy2*dy2);
	  }
          
	  for (k=0; k<nb_clusters; k++)
	    if (k != i) {
	      // we study the move of jNode from clusters[i] to clusters[k]
              
             if (jNode.fake)
               dimbalance = 0;
             else
               dimbalance = dimbalance1;
              
              if (!jNode.fake)
                dimbalance += tdimbalance2[k];
              
	      dcutlinks = 0;
	      for (l=0; l<jNode.links.size(); l++) {
		lNode = (Node)jNode.links.elementAt(l);
		linkweight = ((MyFloat)jNode.roles.elementAt(l)).val;
                dx1 = lNode.master.x - clusters[k].x;
                dy1 = lNode.master.y - clusters[k].y;
                dcutlinks += linkweight * (Math.sqrt(dx1*dx1 + dy1*dy1) - tdcutlinks2[l]);
	      }
	      if ((temp_score = dimbalance + alpha*dcutlinks) < best_score) {
		best_score = temp_score;
		best_k = k;
	      }
	    } // for k
	  if (best_score < -0.00001) {
	    // do the move !
	    jNode.master = clusters[best_k];
	    clusters[best_k].nodes.addElement(jNode);
	    ag.nodes.removeElementAt(j);
            if (!jNode.fake) {
              clusters[best_k].nbreal++;
              ag.nbreal--;
              if (use_keywords)
	        dimbalance1 = balance_cost(ag.nbreal-1,moyna) - balance_cost(ag.nbreal,moyna);
              else
	        dimbalance1 = balance_cost(ag.nodes.size()-1,moyna) - balance_cost(ag.nodes.size(),moyna);
              if (use_keywords)
                tdimbalance2[i] = balance_cost(clusters[i].nbreal+1,moyna) - balance_cost(clusters[i].nbreal,moyna);
              else
                tdimbalance2[i] = balance_cost(clusters[i].nodes.size()+1,moyna) - balance_cost(clusters[i].nodes.size(),moyna);
              if (use_keywords)
                tdimbalance2[best_k] = balance_cost(clusters[best_k].nbreal+1,moyna) - balance_cost(clusters[best_k].nbreal,moyna);
              else
                tdimbalance2[best_k] = balance_cost(clusters[best_k].nodes.size()+1,moyna) - balance_cost(clusters[best_k].nodes.size(),moyna);
            }
            
	    j--;
	    change = true;
	    //System.out.println("Moved "+jNode.url+" from agent "+i+" to agent "+best_k+"(score="+best_score+")");
	    change_best_score = best_score;
	  }
	} // for j
      } // for i
      
      nb_loops_for_bug++;
      if (nb_loops_for_bug > 5000) {
	System.err.println("Too many loops in descent... is there a bug ?");
	System.err.println("best_score was: " + change_best_score);
	change = false;
      }
    } // while change
  }

  // cost evaluation
  private float eval_cost(Vector allNodes, Cluster[] clusters, int nb_clusters, float alpha) {
    float moyna;
    float dimbalance,dcutlinks;
    int i,j;
    Node iNode,jNode;
    float agdist; // distance between the clusters on the grid

    if (use_keywords) {
      // do not count keyword (fake) nodes for the balance
      moyna = calc_moyna(allNodes, nb_clusters);
      dimbalance = 0;
      for (i=0; i<nb_clusters; i++)
        dimbalance += balance_cost(clusters[i].nbreal, moyna);
    } else {
      moyna = ((float)allNodes.size()) / nb_clusters;
      dimbalance = 0;
      for (i=0; i<nb_clusters; i++)
        //dimbalance += Math.abs(clusters[i].nodes.size() - moyna);
        dimbalance += balance_cost(clusters[i].nodes.size(), moyna);
    }
    
    dcutlinks = 0;
    for (i=0; i<allNodes.size(); i++)
      ((Node)allNodes.elementAt(i)).mark = false;
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      for (j=0; j<iNode.links.size(); j++) {
	jNode = (Node)iNode.links.elementAt(j);
	if ((!jNode.mark) && (iNode.master != jNode.master)) {
	  agdist = (float)Math.sqrt(Math.pow(iNode.master.x - jNode.master.x, 2) + Math.pow(iNode.master.y - jNode.master.y, 2));
	  dcutlinks += ((MyFloat)iNode.roles.elementAt(j)).val * agdist;
	}
      }
      iNode.mark = true;
    }
    return(dimbalance + alpha*dcutlinks);
  }

  public void noisingPartitioning(Vector allNodes, Cluster[] clusters, float init_alpha) {
    int i,j,k;
    int ind;
    Node iNode, jNode;
    Cluster ag;
    //int w;
    Vector[] orig_weights;
    Vector vr;
    Cluster[] best_clusters;
    float MAX_RATE = (float)5;
    float MIN_RATE = (float)1.1;
    int MAX_ITER = 100;
    float rate, drate;
    int counter;
    float best_cost,current_cost;
    int STOP_COUNT = 15;
    float z;
    float ALPHA;
    if (init_alpha != 0)
      ALPHA = init_alpha;
    else {
      if (use_keywords)
        ALPHA = (float)0.17;
      else
        ALPHA = (float)0.2;
    }
    int nb_clusters;

    System.out.println("noising partioning");
    nb_clusters = DIMX * DIMY;
    if (allNodes.size() < nb_clusters) {
      System.err.println("allNodes.size() < nb_clusters !!!");
      return;
    }
    
    // checks if all links are to nodes in allNodes
    // (should be useless here, but who knows...)
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      for (j=0; j<iNode.links.size(); j++) {
	jNode = (Node)iNode.links.elementAt(j);
	if (jNode == null) {
	  System.err.println("Null link !!!");
	  iNode.links.removeElementAt(j);
	  j--;
	} else if (!allNodes.contains(jNode)) {
          System.err.println("Node "+jNode.ident+" not in allNodes!!!");
          iNode.links.removeElementAt(j);
          j--;
        }
      }
    }
    
    System.out.println("creating clusters and place them randomly on the map");
    for (i=0; i<allNodes.size(); i++)
      ((Node)allNodes.elementAt(i)).mark = false;
    for (i=0; i<nb_clusters; i++) {
      ind = randomIntBetween(0, allNodes.size()-1);
      while (((Node)allNodes.elementAt(ind)).mark) // I suppose nb_clusters << allNodes.size()
	ind = randomIntBetween(0, allNodes.size()-1);
      clusters[i] = new Cluster((Node)allNodes.elementAt(ind));
      // place randomly the clusters:
      clusters[i].x = (int)Math.round((((float)i)/DIMX - (int)(i/DIMX))*DIMX);
      clusters[i].y = (int)Math.round(i/DIMX);
    }

    System.out.println("linking randomly nodes and clusters");
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      if (!iNode.mark) {
	ind = randomIntBetween(0, nb_clusters - 1);
        iNode.mark = true;
	iNode.master = clusters[ind];
	clusters[ind].nodes.addElement(iNode);
      }
    }
    if (use_keywords)
      for (i=0; i<nb_clusters; i++)
        clusters[i].calc_nbreal();

    //check links symetry
    System.out.println("checking links symetry");
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      for (j=0; j<iNode.links.size(); j++) {
	jNode = (Node)iNode.links.elementAt(j);
	if (jNode == null) {
	  System.err.println("Null link !!!");
	  iNode.links.removeElementAt(j);
	  j--;
	} else {
	  k = jNode.links.indexOf(iNode);
	  if (k == -1) {
	    //System.err.println("No link from "+jNode.ident+" to "+iNode.ident+", but one in the other direction !!! ...adding the link...");
	    jNode.links.addElement(iNode);
	    jNode.roles.addElement(new MyFloat(((MyFloat)iNode.roles.elementAt(j)).val));
	  } else if (((MyFloat)iNode.roles.elementAt(j)).val != ((MyFloat)jNode.roles.elementAt(k)).val) {
	    System.err.println("Links with different weight between "+jNode.ident+" and "+iNode.ident+" ...changing the weight...");
	    ((MyFloat)jNode.roles.elementAt(k)).val = ((MyFloat)iNode.roles.elementAt(j)).val;
	  }
	}
      }
    }

    //check links unicity
    System.out.println("checking links unicity");
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      for (j=0; j<iNode.links.size(); j++) {
	jNode = (Node)iNode.links.elementAt(j);
	for (k=j+1; k<iNode.links.size(); k++)
	  if (jNode == (Node)iNode.links.elementAt(k)) {
	    System.err.println("Link duplicated from "+iNode.ident+" to "+jNode.ident+" ...deleting one...");
	    iNode.links.removeElementAt(k);
	    iNode.roles.removeElementAt(k);
	    k--;
	  }
      }
    }

    // create original data save
    System.out.println("creating original data save");
    orig_weights = new Vector[allNodes.size()];
    for (i=0; i<allNodes.size(); i++) {
      vr = ((Node)allNodes.elementAt(i)).roles;
      orig_weights[i] = new Vector();
      for (j=0; j<vr.size(); j++)
	orig_weights[i].addElement(new MyFloat(((MyFloat)vr.elementAt(j)).val));
    }

    rate = MAX_RATE;
    drate = (MAX_RATE - MIN_RATE)/MAX_ITER;
    counter = 0;

    descent(allNodes, clusters, nb_clusters, ALPHA);

    // record the best solution
    System.out.println("record the best solution");
    best_clusters = new Cluster[nb_clusters];
    for (i=0; i<nb_clusters; i++) {
      best_clusters[i] = new Cluster();
      best_clusters[i].centerURL = new String(clusters[i].centerURL);
      best_clusters[i].title = new String(clusters[i].title);
      best_clusters[i].nodes = (Vector)clusters[i].nodes.clone();
      best_clusters[i].nbreal = clusters[i].nbreal;
    }

    best_cost = eval_cost(allNodes, clusters, nb_clusters, ALPHA);
    System.out.println("Current cost: "+best_cost);
    clusframe.progress("Current cost: "+best_cost);

    while ((counter < STOP_COUNT) && (rate >= MIN_RATE)) {
      // noise the data set
      for (i=0; i<allNodes.size(); i++)
	((Node)allNodes.elementAt(i)).mark = false;
      for (i=0; i<allNodes.size(); i++) {
	iNode = (Node)allNodes.elementAt(i);
	vr = iNode.roles;
	for (j=0; j<vr.size(); j++) {
	  jNode = (Node)iNode.links.elementAt(j);
	  if (!jNode.mark) {
	    z = randomFloatBetween(1, rate);
	    if (randomIntBetween(0,1) == 0)
	      z = 1/z;
	    ((MyFloat)vr.elementAt(j)).val *= z;
	    k = jNode.links.indexOf(iNode);
	    if (k == -1) {
	      System.err.println("No link from "+jNode.url+" to "+iNode.url+", but one in the other direction !!!");
	      return;
	    } else
	      ((MyFloat)jNode.roles.elementAt(k)).val = ((MyFloat)vr.elementAt(j)).val;
	  }
	}
	iNode.mark = true;
      }

      if (counter/2 == ((float)counter)/2) {
	// use best solution
	for (i=0; i<nb_clusters; i++) {
	  clusters[i].centerURL = new String(best_clusters[i].centerURL);
	  clusters[i].title = new String(best_clusters[i].title);
	  clusters[i].nodes = (Vector)best_clusters[i].nodes.clone();
          clusters[i].nbreal = best_clusters[i].nbreal;
	  for (j=0; j<clusters[i].nodes.size(); j++)
	    ((Node)clusters[i].nodes.elementAt(j)).master = clusters[i];
	}
	descent(allNodes, clusters, nb_clusters, ALPHA);
      } else {
	// use current solution
	descent(allNodes, clusters, nb_clusters, ALPHA);
      }
      // use original data
      for (i=0; i<allNodes.size(); i++) {
	vr = ((Node)allNodes.elementAt(i)).roles;
	for (j=0; j<vr.size(); j++)
	  ((MyFloat)vr.elementAt(j)).val = ((MyFloat)orig_weights[i].elementAt(j)).val;
      }
      descent(allNodes, clusters, nb_clusters, ALPHA);
      current_cost = eval_cost(allNodes, clusters, nb_clusters, ALPHA);
      System.out.println("Current cost: "+current_cost);
      clusframe.progress("Current cost: "+current_cost);
      if (current_cost < best_cost) {
	System.out.println("This is an improvement !");
	best_cost = current_cost;
	// record best solution
	for (i=0; i<nb_clusters; i++) {
	  best_clusters[i].centerURL = new String(clusters[i].centerURL);
	  best_clusters[i].title = new String(clusters[i].title);
	  best_clusters[i].nodes = (Vector)clusters[i].nodes.clone();
          best_clusters[i].nbreal = clusters[i].nbreal;
	}
	counter = 0;
      } else
	counter++;
      rate -= drate;
    }

    // use best solution
    for (i=0; i<nb_clusters; i++) {
      clusters[i].centerURL = new String(best_clusters[i].centerURL);
      clusters[i].title = new String(best_clusters[i].title);
      clusters[i].nodes = (Vector)best_clusters[i].nodes.clone();
      clusters[i].nbreal = best_clusters[i].nbreal;
      for (j=0; j<clusters[i].nodes.size(); j++)
	((Node)clusters[i].nodes.elementAt(j)).master = clusters[i];
    }

    System.out.println("Best cost: "+best_cost);
    clusframe.progress("Best cost: "+best_cost);
    //System.out.println(dimbalance+" + "+ALPHA+"*"+dcutlinks+" = "+(dimbalance+ALPHA*dcutlinks));
    System.out.println("");

    // delete links between nodes of different clusters, and replace
    // them by links between the clusters
    // EXCEPT for the keyword nodes
    for (i=0; i<allNodes.size(); i++) {
      iNode = (Node)allNodes.elementAt(i);
      for (j=0; j<iNode.links.size(); j++) {
	jNode = (Node)iNode.links.elementAt(j);
        if ((!jNode.fake) && (!iNode.fake))
	  if (iNode.master != jNode.master) {
	    //w = Math.round(((MyFloat)iNode.roles.elementAt(j)).val);
	    //iNode.master.addLinks(jNode.master, w);
	    //jNode.master.addLinks(iNode.master, w);
	    iNode.links.removeElementAt(j);
	    iNode.roles.removeElementAt(j);
	    j--;
	  }
      }
    }
  }

  public void recursPartitioning(Vector nodes, Cluster[] clusters, Cluster[] precclusters, int zoom, float init_alpha) {
    Vector selectedNodes;
    int i,j,k,l;
    int nbreal;
    boolean haslink;
    Node jNode, kNode, newNode;
    
    noisingPartitioning(nodes, clusters, init_alpha);
    for (i=0; i<clusters.length; i++)
      clusters[i].precClusters = precclusters;
    if (zoom < MAX_ZOOM) {
      for (i=0; i<clusters.length; i++) {
	selectedNodes = new Vector();
        nbreal = 0;
	for (j=0; j<nodes.size(); j++)
	  if (((Node)nodes.elementAt(j)).master == clusters[i]) {
            if (!((Node)nodes.elementAt(j)).fake) {
              nbreal++;
	      selectedNodes.addElement(nodes.elementAt(j));
            }
          }
	if (nbreal > MAX_NB_AGENTS) {
          if (use_keywords) {
          // add all useful keyword nodes
            for (j=0; j<nodes.size(); j++) {
              jNode = (Node)nodes.elementAt(j);
              if ((jNode.fake) && (jNode.master == clusters[i])) {
                newNode = jNode.clone2();
                for (k=0; k<jNode.links.size(); k++) {
                  kNode = (Node)jNode.links.elementAt(k);
                  if (kNode.master == clusters[i]) {
                    newNode.links.addElement(kNode);
                    newNode.roles.addElement(jNode.roles.elementAt(k));
                    for (l=0; l<kNode.links.size(); l++)
                      if ((Node)kNode.links.elementAt(l) == jNode) {
                        kNode.links.setElementAt(newNode, l);
                      }
                    jNode.links.removeElementAt(k);
                    jNode.roles.removeElementAt(k);
                    k--;
                  }
                }
                selectedNodes.addElement(newNode);
              } else if ((jNode.fake) && (jNode.master != clusters[i])) {
                for (k=0, haslink=false; (!haslink) && (k<jNode.links.size()); k++)
                  if (((Node)jNode.links.elementAt(k)).master == clusters[i])
                    haslink = true;
                if (haslink) {
                  newNode = jNode.clone2();
                  newNode.master = clusters[i];
                  for (k=0; k<jNode.links.size(); k++) {
                    kNode = (Node)jNode.links.elementAt(k);
                    if (kNode.master == clusters[i]) {
                      newNode.links.addElement(kNode);
                      newNode.roles.addElement(jNode.roles.elementAt(k));
                      for (l=0; l<kNode.links.size(); l++)
                        if ((Node)kNode.links.elementAt(l) == jNode) {
                          kNode.links.setElementAt(newNode, l);
                        }
                      jNode.links.removeElementAt(k);
                      jNode.roles.removeElementAt(k);
                      k--;
                    }
                  }
                  selectedNodes.addElement(newNode);
                } // if (haslink)
              } // else if ((jNode.fake) && (jNode.master != clusters[i]))
            } // for j
          } // if (use_keywords)
	  clusters[i].subClusters = new Cluster[MAX_NB_AGENTS];
	  recursPartitioning(selectedNodes, clusters[i].subClusters, clusters, zoom+1, init_alpha);
	}
      }
    }
  }
}
