/*
    Copyright (C) 2004  Damien Guillaume
    
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package clustering;

import java.lang.*;
import java.util.Vector;


/**
 * A class used by Clustering to represent an cluster, grouping
 * different nodes (documents).
 *
 * @version     2.0, 15 April 2004
 * @author      Damien Guillaume
 * @see         clustering.Clustering
 * @see         clustering.Node
 */

public class Cluster {

  String centerURL;
  String title;
  Vector nodes; // sorted sizable array of Node
  int x,y;
  Cluster[] subClusters; // Clusters of the next level
  Cluster[] precClusters; // Clusters of the previous level
  int nbreal; // number of real nodes (not fake), for NoisingPartitioning

  public Cluster() {
    centerURL = "";
    nodes = new Vector();
    x = -1;
    y = -1;
    title = "";
    subClusters = null;
    precClusters = null;
    nbreal = 0;
  }

  public Cluster(Node aNode) {
    centerURL = aNode.url;
    nodes = new Vector();
    nodes.addElement(aNode);
    x = -1;
    y = -1;
    aNode.mark = true;
    aNode.master = this;
    if (!aNode.title.equals(""))
      title = aNode.title;
    else
      title = aNode.ident;
  }

  /**
   * search a Node by dichotomy, in a sorted Vector of Nodes, with the url field
   */
  static public ResiField iField(Vector v, String s) {
    int a=0;
    int b=v.size()-1;
    int m;
    int r;
    ResiField res;

    res = new ResiField();
    if (b == -1) {
      res.ind = 0;
      res.found = false;
      return(res);
    }
    while (a+1 < b) {
      m = (a+b)/2;
      r = ((Node)v.elementAt(m)).url.compareTo(s);
      if (r < 0)
	a = m;
      else if (r > 0)
	b = m;
      else {
	res.ind = m;
	res.found = true;
	return(res);
      }
    }
    r = ((Node)v.elementAt(a)).url.compareTo(s);
    if (r == 0) {
      res.ind = a;
      res.found = true;
      return(res);
    } else if (r > 0) {
      res.ind = a;
      res.found = false;
      return(res);
    } else {
      r = ((Node)v.elementAt(b)).url.compareTo(s);
      if (r == 0) {
	res.ind = b;
	res.found = true;
	return(res);
      } else if (r > 0) {
	res.ind = b;
	res.found = false;
	return(res);
      } else {
	res.ind = b+1;
	res.found = false;
	return(res);
      }
    }
  }

  public static String inv_role(String role) {
    if (role.equals(Clustering.HAS_FOR_REF))
      return(Clustering.IS_REF_OF);
    if (role.equals(Clustering.IS_REF_OF))
      return(Clustering.HAS_FOR_REF);
    if (role.equals(Clustering.BEING_STUDIED_BY))
      return(Clustering.IS_STUDYING);
    if (role.equals(Clustering.IS_STUDYING))
      return(Clustering.BEING_STUDIED_BY);
    if (role.equals(Clustering.IS_INCLUDED_IN))
      return(Clustering.INCLUDES);
    if (role.equals(Clustering.INCLUDES))
      return(Clustering.IS_INCLUDED_IN);
    if (role.equals(Clustering.OTHER_DATA))
      return(Clustering.OTHER_DATA);
    if (role.equals(Clustering.HAS_FOR_AUTHOR))
      return(Clustering.IS_AUTHOR_OF);
    if (role.equals(Clustering.IS_AUTHOR_OF))
      return(Clustering.HAS_FOR_AUTHOR);
    if (role.equals(Clustering.IS_AN_IMAGE_OF))
      return(Clustering.HAS_FOR_IMAGE);
    if (role.equals(Clustering.HAS_FOR_IMAGE))
      return(Clustering.IS_AN_IMAGE_OF);
    return("");
  }

  /**
   * nb nodes, without the fake nodes, for NoisingPartitioning
   */
  public void calc_nbreal() {
    int i, nb=0;
    
    for (i=0; i<nodes.size(); i++)
      if (!((Node)nodes.elementAt(i)).fake)
        nb++;
    
    nbreal = nb;
  }
  
}
