/*
Calcotron - Calculateur Scientifique

Copyright (C) 2003-2005 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformément aux dispositions de la Licence Publique Générale GNU, telle que publiée par la Free Software Foundation ; version 2 de la licence, ou encore (à votre choix) toute version ultérieure.

Ce programme est distribué dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans même la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de détail, voir la Licence Publique Générale GNU .

Vous devez avoir reçu un exemplaire de la Licence Publique Générale GNU en même temps que ce programme ; si ce n'est pas le cas, écrivez à la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package calcotron;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.MemoryImageSource;
import java.lang.reflect.Method;
import java.net.*;
import java.util.Vector;

// attention, ce fichier est encodé en UTF-8
class CalcoFrame extends Frame implements KeyListener, ComponentListener {

    private boolean appliquette;
    private Font policeBoutons = new Font("serif", Font.BOLD, 12);
    private Font policeExp = new Font("serif", Font.PLAIN, 9);
    private Vector trucs;
    private MenuConst mconstantes;
    private Affichage affichage;
    private String valeurAffichee;
    private String expressionCachee;
    private double valeur1;
    private double valeur2;
    private boolean edit;
    private Fonction opCourante;
    private double memoire;
    private boolean modeExpressions;
    private String degrad;
    private Image fond;
    private Image logo;
    private final Color grisclair1 = new Color(127,127,127);
    private final Color grisclair2 = new Color(137,137,137);
    private final Color grisclair3 = new Color(142,142,142);
    private final Color grisclair4 = new Color(240,240,240);
    private final Color grisclair5 = new Color(245,245,245);
    private final Color grisfonce1 = new Color(50,50,50);
    private final Color grisfonce2 = new Color(100,100,100);
    
    public static void main(String[] args) {
        CalcoFrame f = new CalcoFrame(false);
        f.setVisible(true);
    }

    public CalcoFrame(boolean appliquette) {
        super("Le Calcotron");
        
        this.appliquette = appliquette;
        valeur2 = 0;
        opCourante = null;
        valeur1 = 0;
        valeurAffichee = "0";
        expressionCachee = "";
        edit = false;
        memoire = 0;
        modeExpressions = false;
        degrad = "rad";
        
        trucs = new Vector();
        trucs.addElement(new Addition());
        trucs.addElement(new Soustraction());
        trucs.addElement(new Multiplication());
        trucs.addElement(new Division());
        trucs.addElement(new Puissance());
        trucs.addElement(new Inverse());
        trucs.addElement(new PuissanceInverse());
        trucs.addElement(new Factorielle());
        trucs.addElement(new M());
        trucs.addElement(new RM());
        trucs.addElement(new Mplus());
        trucs.addElement(new Pourcent());
        trucs.addElement(new Expressions());
        trucs.addElement(new Lettre(','));
        trucs.addElement(new DMS());
        trucs.addElement(new SigneInverse());
        trucs.addElement(new DegRad());
        trucs.addElement(new Del());
        trucs.addElement(new C());
        trucs.addElement(new AC());
        trucs.addElement(new Sinus());
        trucs.addElement(new ArcSinus());
        trucs.addElement(new Cosinus());
        trucs.addElement(new ArcCosinus());
        trucs.addElement(new Tangente());
        trucs.addElement(new ArcTangente());
        trucs.addElement(new Exponentielle());
        trucs.addElement(new LogarithmeNeperien());
        trucs.addElement(new Exp10());
        trucs.addElement(new LogarithmeDecimal());
        trucs.addElement(new Carre());
        trucs.addElement(new RacineCarree());
        trucs.addElement(new Lettre('('));
        trucs.addElement(new Lettre(')'));
        trucs.addElement(new Lettre('0'));
        trucs.addElement(new Lettre('1'));
        trucs.addElement(new Lettre('2'));
        trucs.addElement(new Lettre('3'));
        trucs.addElement(new Lettre('4'));
        trucs.addElement(new Lettre('5'));
        trucs.addElement(new Lettre('6'));
        trucs.addElement(new Lettre('7'));
        trucs.addElement(new Lettre('8'));
        trucs.addElement(new Lettre('9'));
        trucs.addElement(new Lettre('.'));
        trucs.addElement(new Lettre('E'));
        trucs.addElement(new Lettre('='));
        
        mconstantes = new MenuConst("const");
        mconstantes.ajouter("π", "π", Math.PI);
        mconstantes.ajouter("e", "e", Math.E);
        
        mconstantes.ajouter("c", "c (m/s)", 2.99792458E8);
        mconstantes.ajouter("G", "G (kg-1 m3 s-2)", 6.67259E-11);
        mconstantes.ajouter("h", "h (J s)", 6.6260755E-34);
        mconstantes.ajouter("k", "k (J/K)", 1.380622E-23);
        
        mconstantes.ajouter("c_e", "c_e (C)", 1.60217733E-19);
        mconstantes.ajouter("m_e", "m_e (kg)", 9.1093897E-31);
        mconstantes.ajouter("m_p", "m_p (kg)", 1.6726231E-27);
        mconstantes.ajouter("m_n", "m_n (kg)", 1.6749286E-27);
        
        mconstantes.ajouter("UA", "UA (m)", 1.49597870691E11);
        mconstantes.ajouter("AL", "AL (m)", 9.463E15);
        mconstantes.ajouter("pc", "pc (m)", 3.0856775807E16);
        
        // données sur les planètes issues de
        // http://portail.imcce.fr/fr/ephemerides/astronomie/Promenade/pages1/19.html
        MenuConst mplanetes = new MenuConst("système solaire");
        MenuConst mlune = new MenuConst("Lune");
        mlune.ajouter("alun", "demi-grand axe (m)", 3.83398E8);
        mlune.ajouter("elun", "excentricité", 0.05555);
        mlune.ajouter("Tlun", "période sid. (j)", 27.3217); // période de révolution sidérale (en jours terrestre)
        mlune.ajouter("dlun", "diamètre eq. (m)", 3.4442E6); // diametre équatorial
        mlune.ajouter("mlun", "masse (kg)", 7.34753E22);
        mplanetes.ajouter(mlune);
        MenuConst msoleil = new MenuConst("Soleil");
        msoleil.ajouter("msol", "masse (kg)", 1.9887973E30);
        msoleil.ajouter("dsol", "diamètre eq. (m)", 1.392E9);
        msoleil.ajouter("Tsol", "température (K)", 5780); // température de surface
        msoleil.ajouter("Lsol", "luminosité (W)", 3.846E26); // luminosité
        mplanetes.ajouter(msoleil);
        MenuConst mmercure = new MenuConst("Mercure");
        mmercure.ajouter("amer", "demi-grand axe (UA)", 0.3870983);
        mmercure.ajouter("emer", "excentricité", 0.20563);
        mmercure.ajouter("Tmer", "période sid. (j)", 87.9693); // période de révolution sidérale (en jours terrestre)
        mmercure.ajouter("dmer", "diamètre eq. (m)", 4.8794E6); // diametre équatorial
        mmercure.ajouter("mmer", "masse (kg)", 3.3018E23); // masse sans compter les satellites
        mplanetes.ajouter(mmercure);
        MenuConst mvenus = new MenuConst("Vénus");
        mvenus.ajouter("aven", "demi-grand axe (UA)", 0.7233298);
        mvenus.ajouter("even", "excentricité", 0.00677);
        mvenus.ajouter("Tven", "période sid. (j)", 224.701);
        mvenus.ajouter("dven", "diamètre eq. (m)", 12.1036E6);
        mvenus.ajouter("mven", "masse (kg)", 4.8685E24);
        mplanetes.ajouter(mvenus);
        MenuConst mterre = new MenuConst("Terre");
        mterre.ajouter("ater", "demi-grand axe (UA)", 1.0000010);
        mterre.ajouter("eter", "excentricité", 0.01671);
        mterre.ajouter("Tter", "période sid. (j)", 365.256);
        mterre.ajouter("dter", "diamètre eq. (m)", 12.75628E6);
        mterre.ajouter("mter", "masse (kg)", 5.9736E24);
        mplanetes.ajouter(mterre);
        MenuConst mmars = new MenuConst("Mars");
        mmars.ajouter("amar", "demi-grand axe (UA)", 1.5236793);
        mmars.ajouter("emar", "excentricité", 0.09340);
        mmars.ajouter("Tmar", "période sid. (j)", 686.986);
        mmars.ajouter("dmar", "diamètre eq. (m)", 6.794E6);
        mmars.ajouter("mmar", "masse (kg)", 6.4185E23);
        mplanetes.ajouter(mmars);
        MenuConst mjupiter = new MenuConst("Jupiter");
        mjupiter.ajouter("ajup", "demi-grand axe (UA)", 5.202603);
        mjupiter.ajouter("ejup", "excentricité", 0.04850);
        mjupiter.ajouter("Tjup", "période sid. (j)", 4.332656E3);
        mjupiter.ajouter("djup", "diamètre eq. (m)", 1.42984E8);
        mjupiter.ajouter("mjup", "masse (kg)", 1.8986E27);
        mplanetes.ajouter(mjupiter);
        MenuConst msaturne = new MenuConst("Saturne");
        msaturne.ajouter("asat", "demi-grand axe (UA)", 9.554909);
        msaturne.ajouter("esat", "excentricité", 0.05555);
        msaturne.ajouter("Tsat", "période sid. (j)", 1.0759404E4);
        msaturne.ajouter("dsat", "diamètre eq. (m)", 1.20536E8);
        msaturne.ajouter("msat", "masse (kg)", 5.6846E26);
        mplanetes.ajouter(msaturne);
        MenuConst muranus = new MenuConst("Uranus");
        muranus.ajouter("aura", "demi-grand axe (UA)", 19.21845);
        muranus.ajouter("eura", "excentricité", 0.04638);
        muranus.ajouter("Tura", "période sid. (j)", 3.0688984E4);
        muranus.ajouter("dura", "diamètre eq. (m)", 5.1118E7);
        muranus.ajouter("mura", "masse (kg)", 8.6831E25);
        mplanetes.ajouter(muranus);
        MenuConst mneptune = new MenuConst("Neptune");
        mneptune.ajouter("anep", "demi-grand axe (UA)", 30.11039);
        mneptune.ajouter("enep", "excentricité", 0.00946);
        mneptune.ajouter("Tnep", "période sid. (j)", 6.0183284E4);
        mneptune.ajouter("dnep", "diamètre eq. (m)", 4.9528E7);
        mneptune.ajouter("mnep", "masse (kg)", 1.0243E26);
        mplanetes.ajouter(mneptune);
        MenuConst mpluton = new MenuConst("Pluton");
        mpluton.ajouter("aplu", "demi-grand axe (UA)", 39.54470);
        mpluton.ajouter("eplu", "excentricité", 0.2490);
        mpluton.ajouter("Tplu", "période sid. (j)", 9.0580232E4);
        mpluton.ajouter("dplu", "diamètre eq. (m)", 2.390E6);
        mpluton.ajouter("mplu", "masse (kg)", 1.238E22);
        mplanetes.ajouter(mpluton);
        mconstantes.ajouter(mplanetes);
        
        mconstantes.ajouter("answer", "ultimate answer", 42);
        
        try { 
            MediaTracker tracker = new MediaTracker (this);
            //logo = getImage(getDocumentBase(), "logo.gif");
            URL url = CalcoFrame.class.getResource("logo.gif");
            logo = Toolkit.getDefaultToolkit().getImage(url);
            tracker.addImage(logo, 0);
            tracker.waitForAll();
        } catch (Exception e) { 
            e.printStackTrace();
        }
        
        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        setLayout(gridbag);
        
        affichage = new Affichage();
        c.fill = GridBagConstraints.BOTH;
        c.insets = new Insets(3,5,3,5);
        c.weighty = 1.0;
        c.gridwidth = GridBagConstraints.REMAINDER;
        ajouterBouton(affichage, gridbag, c);
        
        c.weightx = 1.0;
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("degrad"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("del"), gridbag, c);
        ajouterBouton(new Bouton("C"), gridbag, c);
        c.insets = new Insets(3,3,3,10);
        ajouterBouton(new Bouton("AC"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new ImageCanvas(logo), gridbag, c);
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("expressions"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("("), gridbag, c);
        ajouterBouton(new Bouton(")"), gridbag, c);
        c.insets = new Insets(3,3,3,10);
        ajouterBouton(new Bouton(","), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new ChoixConstantes(mconstantes), gridbag, c);
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("RM"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("M"), gridbag, c);
        ajouterBouton(new Bouton("Mplus"), gridbag, c);
        //c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,10);
        c.gridx = 3;
        c.gridy = 3;
        ajouterBouton(new Bouton("%"), gridbag, c);
        
        c.gridx = 0;
        c.gridy = 4;
        c.gridwidth = 1;
        c.insets = new Insets(3,5,10,3);
        ajouterBouton(new Bouton("puissanceInverse"), gridbag, c);
        c.gridx = GridBagConstraints.RELATIVE;
        c.insets = new Insets(3,3,10,3);
        ajouterBouton(new Bouton("fact"), gridbag, c);
        ajouterBouton(new Bouton("dms"), gridbag, c);
        c.insets = new Insets(3,3,10,10);
        ajouterBouton(new Bouton("signeInverse"), gridbag, c);
        c.insets = new Insets(8,3,5,3);
        ajouterBouton(new Bouton("sin"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(8,3,5,5);
        ajouterBouton(new Bouton("asin"), gridbag, c);
        c.gridy = GridBagConstraints.RELATIVE;
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("^"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("inverse"), gridbag, c);
        ajouterBouton(new Bouton("/"), gridbag, c);
        c.insets = new Insets(3,3,3,10);
        ajouterBouton(new Bouton("*"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("cos"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new Bouton("acos"), gridbag, c);
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("7"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("8"), gridbag, c);
        ajouterBouton(new Bouton("9"), gridbag, c);
        c.insets = new Insets(3,3,3,10);
        ajouterBouton(new Bouton("-"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("tan"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new Bouton("atan"), gridbag, c);
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("4"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("5"), gridbag, c);
        ajouterBouton(new Bouton("6"), gridbag, c);
        c.insets = new Insets(3,3,3,10);
        ajouterBouton(new Bouton("+"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("exp"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new Bouton("ln"), gridbag, c);
        
        c.gridwidth = 1;
        c.insets = new Insets(3,5,3,3);
        ajouterBouton(new Bouton("1"), gridbag, c);
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("2"), gridbag, c);
        ajouterBouton(new Bouton("3"), gridbag, c);
        c.gridheight = 2;
        c.gridx = 3;
        c.gridy = 8;
        c.insets = new Insets(3,3,5,10);
        ajouterBouton(new Bouton("="), gridbag, c);
        c.gridheight = 1;
        c.gridx = GridBagConstraints.RELATIVE;
        c.gridy = GridBagConstraints.RELATIVE;
        c.insets = new Insets(3,3,3,3);
        ajouterBouton(new Bouton("exp10"), gridbag, c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,3,5);
        ajouterBouton(new Bouton("log"), gridbag, c);
        
        c.gridx = 0;
        c.gridy = 9;
        c.gridwidth = 1;
        c.insets = new Insets(3,5,5,3);
        ajouterBouton(new Bouton("0"), gridbag, c);
        c.insets = new Insets(3,3,5,3);
        c.gridx = GridBagConstraints.RELATIVE;
        ajouterBouton(new Bouton("."), gridbag, c);
        c.insets = new Insets(3,3,5,5);
        ajouterBouton(new Bouton("E"), gridbag, c);
        c.gridx = 4;
        c.insets = new Insets(3,3,5,3);
        ajouterBouton(new Bouton("sqr"), gridbag, c);
        c.gridx = GridBagConstraints.RELATIVE;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(3,3,5,5);
        ajouterBouton(new Bouton("sqrt"), gridbag, c);
        
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                if (CalcoFrame.this.appliquette)
                    dispose();
                else
                    System.exit(0);
            }
        });
        
        addKeyListener(this);
        addComponentListener(this);
        
        //setSize(new Dimension(227, 341)); // pour éviter un bug sous Java 1.3
        pack();
        
        Dimension dim = getSize();
        
        Dimension ecran = getToolkit().getScreenSize();
        setLocation((ecran.width - dim.width)/2, (ecran.height - dim.height)/2);
        
        fond = creerFond(dim.width, dim.height);
        validate();
    }
    
    public void ajouterBouton(Component comp, GridBagLayout g, GridBagConstraints c) {
        g.setConstraints(comp, c);
        comp.addKeyListener(this); // Java bug parade 4150021
        add(comp);
    }
    
    public String versString(double v) {
        int precision = 10;
        int limitesci = 3; // décimales avant de passer en mode sci
        
        if (v == 0)
            return("0");
        String sj = ""+v;
        if ("Infinity".equals(sj) || "-Infinity".equals(sj) || "NaN".equals(sj))
            return(""+v);
        //System.out.print(v);
        String signe;
        if (v >= 0)
            signe = "";
        else {
            v = -v;
            signe = "-";
        }
        int exposant = (int)Math.floor(Math.log(v)/Math.log(10));
        //System.out.print(" (exp="+exposant+")");
        //double exparr = Math.pow(10, -(exposant - precision));
        //v = Math.rint(v*exparr)/exparr; // arrondi
        double exparr = Math.pow(10, precision);
        double mantisse = v / Math.pow(10, exposant);
        //System.out.print(" ("+Math.rint(mantisse*exparr)/exparr+")");
        //System.out.print(" ("+mantisse+"->");
        mantisse = Math.rint(mantisse*exparr)/exparr;
        //System.out.print(mantisse+")");
        if (mantisse >= 10) {
            exposant += 1;
            mantisse = v / Math.pow(10, exposant);
            mantisse = Math.rint(mantisse*exparr)/exparr;
        }
        String sv = "";
        int chiffre = (int)Math.floor(mantisse);
        sv += chiffre + ".";
        for (int i=0; i<precision-1; i++) {
            //System.out.print(" ("+mantisse+"->");
            mantisse = (mantisse - chiffre)*10;
            double exparr2 = Math.pow(10, precision-i);
            mantisse = Math.rint(mantisse*exparr)/exparr;
            //System.out.print(mantisse+")");
            chiffre = (int)Math.floor(mantisse);
            sv += chiffre;
        }
        mantisse = (mantisse - chiffre)*10;
        chiffre = (int)Math.round(mantisse);
        sv += chiffre;
        char dernier = sv.charAt(sv.length()-1);
        while (dernier == '0') {
            sv = sv.substring(0, sv.length()-1);
            dernier = sv.charAt(sv.length()-1);
        }
        if (exposant != 0) {
            if (exposant > 0 && exposant < limitesci) {
                String sd = sv.substring(0,1);
                if (sv.length()-2 < exposant) {
                    // transformation de 3.2E2 en 320
                    if (sv.length() > 2)
                        sd += sv.substring(2,sv.length());
                    for (int i=0; i<exposant-(sv.length()-2); i++)
                        sd += '0';
                } else {
                    // transformation de 3.245E2 en 324.5
                    if (sv.length() < exposant+2)
                        sd += sv.substring(2);
                    else
                        sd += sv.substring(2,exposant+2) + "." + sv.substring(exposant+2);
                }
                sv = sd;
            } else if (exposant < 0 && exposant > -limitesci) {
                // transformation de 3.2E-2 en 0.032
                String sd = "0.";
                for (int i=0; i<-exposant-1; i++)
                    sd += '0';
                sv = sd + sv.charAt(0) + sv.substring(2);
            } else
                sv += "E" + exposant;
        }
        if (sv.charAt(sv.length()-1) == '.')
            sv = sv.substring(0, sv.length()-1);
        
        sv = signe + sv;
        
        //System.out.println(" -> " + sv);
        return(sv);
    }
    
    public void operation(Truc truc) {
        if (truc instanceof Lettre && "=".equals(((Lettre)truc).getNom())) {
            if (modeExpressions) {
                Double valeur = calculer(valeurAffichee);
                if (valeur == null)
                    valeurAffichee = "erreur";
                else {
                    valeur1 = valeur.doubleValue();
                    valeurAffichee = versString(valeur1);
                    expressionCachee = "" + valeur1;
                }
                valeur2 = 0;
                opCourante = null;
                edit = false;
            } else {
                try {
                    if (edit) {
                        valeur1 = (new Double(valeurAffichee)).doubleValue();
                        expressionCachee += valeur1;
                        if (opCourante != null && !opCourante.estOperateur())
                            expressionCachee += ')';
                    }
                    if (opCourante != null) {
                        Double valeur = calculer(expressionCachee);
                        if (valeur == null)
                            valeurAffichee = "erreur";
                        else {
                            valeur1 = valeur.doubleValue();
                            valeurAffichee = versString(valeur1);
                        }
                    }
                    expressionCachee = "" + valeur1;
                    valeur2 = 0;
                    opCourante = null;
                    edit = false;
                } catch (NumberFormatException ex) {
                    Toolkit.getDefaultToolkit().beep();
                    System.err.println("NumberFormatException: " + ex.getMessage());
                }
            }
        } else if (truc instanceof Lettre) {
            String nom = ((Lettre)truc).getNom();
            if (!modeExpressions && ")".equals(nom) && edit) {
                try {
                    valeur1 = (new Double(valeurAffichee)).doubleValue();
                    expressionCachee += valeur1;
                } catch (NumberFormatException ex) {
                    Toolkit.getDefaultToolkit().beep();
                    System.err.println("NumberFormatException: " + ex.getMessage());
                }
            }
            if (!edit) {
                valeurAffichee = "";
                if (opCourante == null)
                    expressionCachee = "";
                edit = true;
            }
            if (!modeExpressions && ("(".equals(nom) || ")".equals(nom))) {
                if (")".equals(nom)) {
                    valeurAffichee = "";
                    edit = false;
                }
                expressionCachee += nom;
            }
            if (modeExpressions || (!"(".equals(nom) && !")".equals(nom)))
                valeurAffichee += nom;
        } else if (truc instanceof Special) {
            ((Special)truc).executer();
        } else if (truc instanceof Fonction) {
            Fonction fct = (Fonction)truc;
            if (modeExpressions) {
                if (!edit && !fct.estOperateur())
                    valeurAffichee = "";
                valeurAffichee += fct.getNom();
                if (!fct.estOperateur())
                    valeurAffichee += "(";
                edit = true;
            } else {
                // cas spécial du - après un E
                if (("-".equals(fct.getNom()) || "signeInverse".equals(fct.getNom())) &&
                        valeurAffichee.endsWith("E") && edit)
                    valeurAffichee += "-";
                else {
                    try {
                        if (edit) {
                            try {
                                valeur1 = (new Double(valeurAffichee)).doubleValue();
                            } catch (NumberFormatException ex) {
                                Toolkit.getDefaultToolkit().beep();
                                System.err.println("NumberFormatException: " + ex.getMessage());
                            }
                        }
                        if (fct.getNbParams() == 0) {
                            valeur1 = fct.evaluer(0, 0);
                            valeurAffichee = versString(valeur1);
                            if (!edit && opCourante == null)
                                expressionCachee = "";
                            edit = true;
                        } else if (fct.getNbParams() == 1) {
                            valeur1 = fct.evaluer(valeur1, 0);
                            valeurAffichee = versString(valeur1);
                            if (!edit)
                                expressionCachee = "" + valeur1;
                        } else {
                            if (fct.estOperateur()) {
                                if (edit)
                                    expressionCachee += valeur1 + fct.getNom();
                                else
                                    expressionCachee += fct.getNom();
                            } else {
                                if (edit)
                                    expressionCachee += fct.getNom() + '(' + valeur1 + ',';
                                else
                                    expressionCachee = fct.getNom() + '(' + expressionCachee + ',';
                            }
                            valeur2 = valeur1;
                            opCourante = fct;
                            edit = false;
                        }
                    } catch (NumberFormatException ex) {
                        System.err.println("NumberFormatException: " + ex.getMessage());
                    }
                }
            }
        }
        affichage.repaint();
    }
    
    public void antialiasing(Graphics g) {
        if ("1.2".compareTo(System.getProperty("java.version")) <= 0) {
            // Java 1.2+
            // pour rester compatible on utilise la reflection
            /* en clair ça donne:
                Graphics2D g2d = (Graphics2D)g;
                g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                    RenderingHints.VALUE_ANTIALIAS_ON);
                g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
                    RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            */
            try {
                // et voilà la version cryptée :)
                Class RenderingHints = Class.forName("java.awt.RenderingHints");
                Object KEY_ANTIALIASING = RenderingHints.getField("KEY_ANTIALIASING").get(null);
                Object KEY_TEXT_ANTIALIASING = RenderingHints.getField("KEY_TEXT_ANTIALIASING").get(null);
                Object VALUE_ANTIALIAS_ON = RenderingHints.getField("VALUE_ANTIALIAS_ON").get(null);
                Object VALUE_TEXT_ANTIALIAS_ON = RenderingHints.getField("VALUE_TEXT_ANTIALIAS_ON").get(null);
                Class Graphics2D = Class.forName("java.awt.Graphics2D");
                Class[] parameterTypes = new Class[2];
                parameterTypes[0] = Class.forName("java.awt.RenderingHints$Key");
                parameterTypes[1] = Object.class;
                Method setRenderingHint = Graphics2D.getMethod("setRenderingHint", parameterTypes);
                Object[] args = new Object[2];
                args[0] = KEY_ANTIALIASING;
                args[1] = VALUE_ANTIALIAS_ON;
                setRenderingHint.invoke(g, args);
                args[0] = KEY_TEXT_ANTIALIASING;
                args[1] = VALUE_TEXT_ANTIALIAS_ON;
                setRenderingHint.invoke(g, args);
            } catch (Exception ex) {
                System.err.println("CalcoFrame: " + ex.getClass().getName() + ": " + ex.getMessage());
            }
        }
    }
    
    class Affichage extends Canvas {
        public void paint (Graphics g) {
            antialiasing(g);
            Dimension vsize = getSize();
            Rectangle r = new Rectangle(vsize.width, vsize.height);
            g.setColor(Color.white);
            g.fillRect(r.x, r.y, r.width, r.height);
            g.setColor(Color.black);
            g.setFont(policeBoutons);
            FontMetrics metrics = g.getFontMetrics();
            int sh = metrics.getHeight();
            int sw = metrics.stringWidth(valeurAffichee);
            g.drawString(valeurAffichee, r.width - sw - 5, (r.height+sh)/2);
            g.setFont(policeExp);
            sh = g.getFontMetrics().getHeight();
            String infos = degrad;
            if (modeExpressions)
                infos += " expr";
            g.drawString(infos, 5, r.height-1);
            if (memoire != 0)
                g.drawString("M", 5, sh+1);
            g.drawRect(r.x, r.y, r.width, r.height);
        }
        public Dimension getMinimumSize() { return(new Dimension(180, 30)); }
        public Dimension getPreferredSize() { return(getMinimumSize()); }
    }
    
    class Bouton extends Component implements MouseListener {
        Truc truc = null;
        boolean appui = false;
        final int esp = 0;
        final int rr = 7;
        
        public Bouton(String nomTruc) {
            truc = chercherTruc(nomTruc);
            if (truc == null)
                System.err.println("opération non trouvée: " + nomTruc);
            else
                truc.setBouton(this);
            addMouseListener(this);
        }
        
        public void paint (Graphics g) {
            if (truc == null)
                return;
            antialiasing(g);
            if (truc.getTitre() == null)
                System.err.println("Le titre est null pour " + truc.getNom() + " !");
            Dimension vsize = getSize();
            Rectangle r = new Rectangle(vsize.width, vsize.height);
            g.setColor(grisfonce2);
            g.fillRoundRect(r.x+1+esp, r.y+1+esp, r.width-2-esp*2, r.height-2-esp*2, rr, rr);
            g.setColor(grisfonce1);
            g.fillRoundRect(r.x+esp, r.y+esp, r.width-2-esp*2, r.height-2-esp*2, rr, rr);
            
            if (appui)
                g.setColor(grisclair3);
            else
                g.setColor(Color.white);
            g.fillRoundRect(r.x+esp+1, r.y+esp+1, r.width-4-esp*2, r.height-4-esp*2, rr, rr);
            if (appui)
                g.setColor(grisclair2);
            else
                g.setColor(grisclair5);
            g.fillRoundRect(r.x+esp+1, r.y+esp+1, r.width-4-esp*2, r.y+esp+10, rr, rr);
            if (appui)
                g.setColor(grisclair1);
            else
                g.setColor(grisclair4);
            g.fillRoundRect(r.x+esp+1, r.y+esp+5, r.width-4-esp*2, r.height-2-esp*2-10, rr, rr);
            g.setColor(Color.black);
            
            String titre = truc.getTitre();
            if ("√".equals(titre)) {
                // ce caractère n'est pas disponible partout...
                int mx = (r.x + r.width)/2;
                int my = (r.y + r.height)/2;
                g.drawLine(mx-9, my, mx-8, my);
                g.drawLine(mx-8, my, mx-4, my+8);
                g.drawLine(mx-4, my+8, mx, my-8);
                g.drawLine(mx, my-8, mx+8, my-8);
                g.drawLine(mx+8, my-8, mx+8, my-7);
            } else {
                g.setFont(policeBoutons);
                FontMetrics metrics = g.getFontMetrics();
                int sh = metrics.getHeight();
                int psup = titre.indexOf("<sup>");
                if (psup == -1) {
                    int sw = metrics.stringWidth(titre);
                    g.drawString(titre, (r.width-2-sw)/2, (r.height-2+sh)/2);
                } else {
                    int psup2 = titre.indexOf("</sup>");
                    String snorm;
                    int sw = 0;
                    String ssup = titre.substring(psup+5, psup2);
                    g.setFont(policeExp);
                    metrics = g.getFontMetrics(); // le premier appel est très lent sur Linux et Windows
                    sw += metrics.stringWidth(ssup);
                    if (psup == 0)
                        snorm = titre.substring(psup2+6);
                    else
                        snorm = titre.substring(0, psup);
                    if ("".equals(snorm))
                        snorm = null;
                    g.setFont(policeBoutons);
                    metrics = g.getFontMetrics();
                    if (snorm != null)
                        sw += metrics.stringWidth(snorm);
                    int iw = 0;
                    if (psup == 0) {
                        g.setFont(policeExp);
                        metrics = g.getFontMetrics();
                        iw += metrics.stringWidth(ssup);
                        g.drawString(ssup, (r.width-sw)/2, (r.height)/2);
                        if (snorm != null) {
                            g.setFont(policeBoutons);
                            g.drawString(snorm, (r.width-sw)/2+iw, (r.height+sh)/2);
                        }
                    } else {
                        g.setFont(policeBoutons);
                        metrics = g.getFontMetrics();
                        iw += metrics.stringWidth(snorm);
                        g.drawString(snorm, (r.width-2-sw)/2, (r.height-2+sh)/2);
                        g.setFont(policeExp);
                        g.drawString(ssup, (r.width-2-sw)/2+iw, (r.height-2)/2);
                    }
                }
            }
        }
        
        public void mouseClicked(MouseEvent e) { }
        
        public void mousePressed(MouseEvent e) {
            activation(true);
        }
        
        public void mouseReleased(MouseEvent e) {
            operation(truc);
            activation(false);
            //CalcoFrame.this.requestFocus();
            // ce requestFocus bloque l'envoi de keyTyped avec Java 1.4.2 sous MacOS X
            // -> on est obligé d'ajouter un KeyListener pour tous les boutons
        }
        
        public void activation(boolean actif) {
            if (appui != actif) {
                appui = actif;
                repaint();
            }
        }
        
        public void mouseEntered(MouseEvent e) { }
        
        public void mouseExited(MouseEvent e) { }
        
        public Dimension getMinimumSize() { return(new Dimension(32+esp*2, 30+esp*2)); }
        public Dimension getPreferredSize() { return(getMinimumSize()); }
    }
    
    class ImageCanvas extends Canvas {
        Image img;
        public ImageCanvas(Image img) {
            this.img = img;
        }
        
        public void paint(Graphics g) {
            Dimension vsize = getSize();
            g.drawImage(img, 0, 0, vsize.width, vsize.height, null);
        }
        
        public Dimension getMinimumSize() { return(new Dimension(60, 30)); }
        public Dimension getPreferredSize() { return(getMinimumSize()); }
    }
    
    public Dimension getMinimumSize() {
        Dimension ms = super.getMinimumSize();
        if (ms.width < 227)
            ms.width = 227;
        if (ms.height < 341)
            ms.height = 341;
        return(ms);
    }
    public Dimension getPreferredSize() { return(getMinimumSize()); }
    
    public void paint(Graphics g) {
        g.drawImage(fond, 0, 0, null);
        super.paint(g);
    }
    
    public void update(Graphics g) {
        paint(g);
    }
    
    public Image creerFond(int w, int h) {
	int[] pix = new int[w * h];
        int nf = 50;
        double[] filtre = new double[nf];
        double[] filtre2 = new double[nf];
        double[] filtre3 = new double[w];
        double somme = 0;
        int dup = 20;
        
        for (int i=0; i<nf; i++)
            filtre[i] = Math.random()/3;
        
        for (int i=0; i<nf; i++) {
            filtre2[i] = nf/2 - Math.abs(i-nf/2);
            somme += filtre2[i];
        }
        
        for (int i=0; i<w; i++)
            filtre3[i] = ((w/2 - Math.abs(i-w/2))/(1.0*w/2))/4 + 0.75;
        
        int index = 0;
        for (int y = 0; y < h; y++) {
            if (y < dup)
                for (int x = 0; x < w; x++) {
                    double d = 0;
                    for (int i=0; i<nf; i++)
                        d += filtre[i]*filtre2[i];
                    int c = (int)Math.round(255*(1-d/somme)*filtre3[x]);
                    pix[index++] = (255 << 24) | (c << 16) | (c << 8) | c;
                    for (int i=0; i<nf-1; i++)
                        filtre[i] = filtre[i+1];
                    filtre[nf-1] = Math.random()/3;
                }
            else
                for (int x = 0; x < w; x++) {
                    int ymod = (int)Math.floor((y/(1.0*dup)-y/dup)*dup);
                    pix[index++] = pix[x + ymod*w];
                }
        }
        return(createImage(new MemoryImageSource(w, h, pix, 0, w)));
    }
    
    public void componentHidden(ComponentEvent e) {
    }

    public void componentMoved(ComponentEvent e) {
    }

    public void componentResized(ComponentEvent e) {
        Dimension dim = getSize();
        fond = creerFond(dim.width, dim.height);
        Dimension prefdim = getPreferredSize();
        double fact = Math.min(((double)dim.width) / prefdim.width, ((double)dim.height) / prefdim.height);
        policeBoutons = new Font("sans-serif", Font.BOLD, (int)Math.round(10*fact));
        policeExp = new Font("sans-serif", Font.PLAIN, (int)Math.round(7*fact));
        repaint();
    }

    public void componentShown(ComponentEvent e) {
    }
    
    public void keyTyped(KeyEvent e) {
        if (e.getKeyChar() == ',')
            touche('.');
        else
            touche(e.getKeyChar());
    }
    
    public void keyPressed(KeyEvent e) {
        int code = e.getKeyCode();
        if (code == KeyEvent.VK_ENTER || code == KeyEvent.VK_UNDEFINED) {
            Truc truc = chercherTruc("=");
            if (truc != null) {
                truc.activation(true);
                operation(truc);
            }
        } else if (code == KeyEvent.VK_BACK_SPACE) {
            Truc truc = chercherTruc("del");
            if (truc != null) {
                truc.activation(true);
                operation(truc);
            }
        } else if (code == KeyEvent.VK_DELETE) {
            Truc truc = chercherTruc("AC");
            if (truc != null) {
                truc.activation(true);
                operation(truc);
            }
        }
        if (e.getKeyChar() != KeyEvent.CHAR_UNDEFINED) {
            Truc truc = chercherTruc("" + e.getKeyChar());
            if (truc != null)
                truc.activation(true);
        }
    }
    
    public void touche(char c) {
        String sc = "" + c;
        Truc truc = chercherTruc(sc);
        if (truc != null)
            operation(truc);
    }
    
    public void keyReleased(KeyEvent e) {
        if (e.getKeyChar() != KeyEvent.CHAR_UNDEFINED) {
            Truc truc = chercherTruc("" + e.getKeyChar());
            if (truc != null)
                truc.activation(false);
        }
        int code = e.getKeyCode();
        if (code == KeyEvent.VK_ENTER || code == KeyEvent.VK_UNDEFINED) {
            Truc truc = chercherTruc("=");
            if (truc != null)
                truc.activation(false);
        } else if (code == KeyEvent.VK_BACK_SPACE) {
            Truc truc = chercherTruc("del");
            if (truc != null)
                truc.activation(false);
        } else if (code == KeyEvent.VK_DELETE) {
            Truc truc = chercherTruc("AC");
            if (truc != null)
                truc.activation(false);
        }
    }
    
    public Truc chercherTruc(String nom) {
        for (int i=0; i<trucs.size(); i++) {
            if (((Truc)trucs.elementAt(i)).getNom().equalsIgnoreCase(nom))
                return((Truc)trucs.elementAt(i));
        }
        return(null);
    }
    
    
    abstract class Truc {
        String nom;
        String titre;
        Bouton bouton;
        public Truc(String nom, String titre) {
            this.nom = nom;
            this.titre = titre;
        }
        public String getNom() {
            return(nom);
        }
        public String getTitre() {
            return(titre);
        }
        public void setBouton(Bouton b) {
            this.bouton = b;
        };
        public void activation(boolean actif) {
            if (bouton != null)
                bouton.activation(actif);
        }
    }
    
    abstract class Fonction extends Truc {
        boolean operateur;
        int nbparams;
        
        public Fonction(String nom, String titre, boolean operateur, int nbparams) {
            super(nom, titre);
            this.operateur = operateur;
            this.nbparams = nbparams;
        }
        public boolean estOperateur() { return(operateur); }
        public int getNbParams() { return(nbparams); }
        public abstract double evaluer(double v1, double v2);
    }
    
    abstract class Special extends Truc {
        public Special(String nom, String titre) {
            super(nom, titre);
        }
        public abstract void executer();
    }
    
    public Fonction obtenirFonction(String nom) {
        Truc truc = chercherTruc(nom);
        if (truc == null) {
            System.err.println("opération non trouvée: " + nom);
            return(null);
        }
        if (!(truc instanceof Fonction)) {
            System.err.println("opération de mauvais type: " + nom);
            return(null);
        }
        return((Fonction)truc);
    }
    
    class Lettre extends Truc {
        public Lettre(char c) {
            super(""+c, ""+c);
        }
    }
    
    class Addition extends Fonction {
        public Addition() {
            super("+", "+", true, 2);
        }
        public double evaluer(double v1, double v2) {
            return(v1 + v2);
        }
    }
    
    class Soustraction extends Fonction {
        public Soustraction() {
            super("-", "-", true, 2);
        }
        public double evaluer(double v1, double v2) {
            return(v1 - v2);
        }
    }
    
    class Multiplication extends Fonction {
        public Multiplication() {
            super("*", "x", true, 2);
        }
        public double evaluer(double v1, double v2) {
            return(v1 * v2);
        }
    }
    
    class Division extends Fonction {
        public Division() {
            super("/", "/", true, 2);
        }
        public double evaluer(double v1, double v2) {
            return(v1 / v2);
        }
    }
    
    class Puissance extends Fonction {
        public Puissance() {
            super("^", "^", true, 2);
        }
        public double evaluer(double v1, double v2) {
            return(Math.pow(v1, v2));
        }
    }
    
    class Inverse extends Fonction {
        public Inverse() {
            super("inverse", "1/x", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(1/v1);
        }
    }
    
    class PuissanceInverse extends Fonction {
        public PuissanceInverse() {
            super("puissanceInverse", "x<sup>1/n</sup>", false, 2);
        }
        public double evaluer(double v1, double v2) {
            return(Math.pow(v1, 1/v2));
        }
    }
    
    class Factorielle extends Fonction {
        public Factorielle() {
            super("fact", "x!", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (Math.floor(v1) == v1) {
                double v = v1;
                while (v1 > 2 && v != Double.POSITIVE_INFINITY) {
                    v *= v1-1;
                    v1--;
                }
                return(v);
            }
            return(v1);
        }
    }
    
    class M extends Fonction {
        public M() {
            super("M", "M", false, 1);
        }
        public double evaluer(double v1, double v2) {
            memoire = v1;
            return(v1);
        }
    }
    
    class Mplus extends Fonction {
        public Mplus() {
            super("Mplus", "M+", false, 1);
        }
        public double evaluer(double v1, double v2) {
            memoire += v1;
            return(v1);
        }
    }
    
    class RM extends Fonction {
        public RM() {
            super("RM", "RM", false, 0);
        }
        public double evaluer(double v1, double v2) {
            return(memoire);
        }
    }
    
    class Pourcent extends Special {
        public Pourcent() {
            super("%", "%");
        }
        public void executer() {
            try {
                if (edit)
                    valeur1 = (new Double(valeurAffichee)).doubleValue();
                if (opCourante != null)
                    valeur1 = opCourante.evaluer(valeur2, valeur1*valeur2/100);
                else
                    valeur1 = valeur1/100;
                valeurAffichee = versString(valeur1);
                expressionCachee = "" + valeur1;
                valeur2 = 0;
                opCourante = null;
                edit = false;
            } catch (NumberFormatException ex) {
                System.err.println("NumberFormatException: " + ex.getMessage());
            }
        }
    }
    
    class Expressions extends Special {
        public Expressions() {
            super("expressions", "expr");
        }
        public void executer() {
            modeExpressions = !modeExpressions;
        }
    }
    
    class SigneInverse extends Fonction {
        public SigneInverse() {
            super("signeInverse", "±", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(-v1);
        }
    }
    
    class DMS extends Special {
        public DMS() {
            super("dms", "° ' \"");
        }
        public void executer() {
            if (valeurAffichee.indexOf('°') != -1) {
                valeurAffichee = versString(valeur1);
            } else {
                try {
                    if (edit)
                        valeur1 = (new Double(valeurAffichee)).doubleValue();
                    String signe = "+";
                    if (valeur1 < 0) {
                        signe = "-";
                        valeur1 = -valeur1;
                    }
                    int degres = (int)Math.floor(valeur1);
                    double dminutes = (valeur1 - degres)*60;
                    int minutes = (int)Math.floor(dminutes);
                    double secondes = Math.rint((dminutes - minutes)*60*100)/100;
                    if (secondes >= 60) {
                        secondes -= 60;
                        minutes += 1;
                    }
                    if (minutes >= 60) {
                        minutes -= 60;
                        degres += 1;
                    }
                    valeurAffichee = signe + degres + "° " + minutes + "' " + secondes + "\"";
                    edit = false;
                    if ("-".equals(signe))
                        valeur1 = -valeur1;
                } catch (NumberFormatException ex) {
                    System.err.println("NumberFormatException: " + ex.getMessage());
                }
            }
        }
    }
    
    class DegRad extends Special {
        public DegRad() {
            super("degrad", "deg");
        }
        public void executer() {
            titre = degrad;
            if ("rad".equals(degrad))
                degrad = "deg";
            else
                degrad = "rad";
        }
    }
    
    class Del extends Special {
        public Del() {
            super("del", "del");
        }
        public void executer() {
            if (valeurAffichee.length() > 0)
                valeurAffichee = valeurAffichee.substring(0, valeurAffichee.length()-1);
        }
    }
    
    class C extends Special {
        public C() {
            super("C", "C");
        }
        public void executer() {
            valeur1 = 0;
            valeurAffichee = "0";
            edit = false;
        }
    }
    
    class AC extends Special {
        public AC() {
            super("AC", "AC");
        }
        public void executer() {
            valeur1 = 0;
            valeurAffichee = "0";
            valeur2 = 0;
            opCourante = null;
            edit = false;
            expressionCachee = "";
        }
    }
    
    class Sinus extends Fonction {
        public Sinus() {
            super("sin", "sin", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.sin(v1));
            else
                return(Math.sin(v1/180*Math.PI));
        }
    }
    
    class ArcSinus extends Fonction {
        public ArcSinus() {
            super("asin", "sin<sup>-1</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.asin(v1));
            else
                return(Math.asin(v1)*180/Math.PI);
        }
    }
    
    class Cosinus extends Fonction {
        public Cosinus() {
            super("cos", "cos", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.cos(v1));
            else
                return(Math.cos(v1/180*Math.PI));
        }
    }
    
    class ArcCosinus extends Fonction {
        public ArcCosinus() {
            super("acos", "cos<sup>-1</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.acos(v1));
            else
                return(Math.acos(v1)*180/Math.PI);
        }
    }
    
    class Tangente extends Fonction {
        public Tangente() {
            super("tan", "tan", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.tan(v1));
            else
                return(Math.tan(v1/180*Math.PI));
        }
    }
    
    class ArcTangente extends Fonction {
        public ArcTangente() {
            super("atan", "tan<sup>-1</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            if (degrad.equals("rad"))
                return(Math.atan(v1));
            else
                return(Math.atan(v1)*180/Math.PI);
        }
    }
    
    class Exponentielle extends Fonction {
        public Exponentielle() {
            super("exp", "e<sup>x</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(Math.exp(v1));
        }
    }
    
    class LogarithmeNeperien extends Fonction {
        public LogarithmeNeperien() {
            super("ln", "ln", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(Math.log(v1));
        }
    }
    
    class Exp10 extends Fonction {
        public Exp10() {
            super("exp10", "10<sup>x</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(Math.pow(10,v1));
        }
    }
    
    class LogarithmeDecimal extends Fonction {
        public LogarithmeDecimal() {
            super("log", "log", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(Math.log(v1)/Math.log(10));
        }
    }
    
    class Carre extends Fonction {
        public Carre() {
            super("sqr", "x<sup>2</sup>", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(v1*v1);
        }
    }
    
    class RacineCarree extends Fonction {
        public RacineCarree() {
            super("sqrt", "√", false, 1);
        }
        public double evaluer(double v1, double v2) {
            return(Math.sqrt(v1));
        }
    }
    
    
    class Constante {
        String nom;
        String titre;
        double valeur;
        
        public Constante(String nom, String titre, double valeur) {
            this.nom = nom;
            this.titre = titre;
            this.valeur = valeur;
        }
        public String getNom() { return(nom); }
        public String getTitre() { return(titre); }
        public double getValeur() { return(valeur); }
    }
    
    // menu récursif pour les constantes
    class MenuConst {
        String titre;
        Vector constantes;
        public MenuConst(String titre) {
            this.titre = titre;
            constantes = new Vector();
        }
        public String getTitre() {
            return(titre);
        }
        public void ajouter(String nom, String titre, double valeur) {
            constantes.addElement(new Constante(nom, titre, valeur));
        }
        public void ajouter(MenuConst menu) {
            constantes.addElement(menu);
        }
        public void creerMenu(Menu m, ActionListener al) {
            for (int i=0; i<constantes.size(); i++) {
                if (constantes.elementAt(i) instanceof Constante) {
                    Constante cst = (Constante)constantes.elementAt(i);
                    MenuItem mi = new MenuItem(cst.getTitre());
                    mi.addActionListener(al);
                    mi.setActionCommand(cst.getNom());
                    m.add(mi);
                } else if (constantes.elementAt(i) instanceof MenuConst) {
                    MenuConst mc = (MenuConst)constantes.elementAt(i);
                    Menu m2 = new Menu(mc.getTitre());
                    mc.creerMenu(m2, al);
                    m.add(m2);
                }
            }
        }
        public Constante getConstante(String nom) {
            for (int i=0; i<constantes.size(); i++) {
                if (constantes.elementAt(i) instanceof Constante) {
                    Constante cst = (Constante)constantes.elementAt(i);
                    if (nom.equals(cst.getNom()))
                        return(cst);
                } else if (constantes.elementAt(i) instanceof MenuConst) {
                    MenuConst mc = (MenuConst)constantes.elementAt(i);
                    Constante cst = mc.getConstante(nom);
                    if (cst != null)
                        return(cst);
                }
            }
            return(null);
        }
    }
    
    class ChoixConstantes extends Component implements ActionListener {
        PopupMenu popup;
        boolean appui=false;
        MenuConst menuconst;
        
        public ChoixConstantes(MenuConst menuconst) {
            super();
            this.menuconst = menuconst;
            popup = new PopupMenu();
            menuconst.creerMenu(popup, this);
            enableEvents(AWTEvent.MOUSE_EVENT_MASK);
            add(popup);
        }
        
        public void paint(Graphics g) {
            antialiasing(g);
            String titre = menuconst.getTitre();
            Dimension vsize = getSize();
            Rectangle r = new Rectangle(vsize.width, vsize.height);
            g.setColor(grisfonce2);
            g.drawRect(r.x+1, r.y+1, r.width-2, r.height-2);
            if (appui)
                g.setColor(Color.black);
            else
                g.setColor(Color.white);
            g.fillRect(r.x, r.y, r.width-2, r.height-2);
            if (appui)
                g.setColor(Color.white);
            else
                g.setColor(Color.black);
            g.setFont(policeBoutons);
            FontMetrics metrics = g.getFontMetrics();
            int sh = metrics.getHeight();
            int sw = metrics.stringWidth(titre);
            g.drawString(titre, (r.width - sw)/2 - 5, (r.height+sh)/2);
            g.setColor(grisfonce1);
            g.drawRect(r.x, r.y, r.width-2, r.height-2);
        }
        
        public void processMouseEvent(MouseEvent e) {
            //if (e.isPopupTrigger())
            /*
            if (appui) {
                appui = false;
                repaint();
            }
            */
            // pb: Java bug 4040614
            if (e.getID() == MouseEvent.MOUSE_PRESSED) {
                /*
                appui = true;
                repaint();
                */
                popup.show(this, e.getX(), e.getY());
            } else
                super.processMouseEvent(e);
        }
        
        public void actionPerformed(ActionEvent e) {
            String nom = e.getActionCommand();
            Constante cst = menuconst.getConstante(nom);
            if (cst == null)
                return;
            if (modeExpressions) {
                if (edit)
                    valeurAffichee += nom;
                else {
                    valeurAffichee = nom;
                    edit = true;
                }
            } else {
                if (opCourante == null)
                    expressionCachee = "";
                valeur1 = cst.getValeur();
                valeurAffichee = versString(valeur1);
                edit = true;
            }
            CalcoFrame.this.requestFocus();
            affichage.repaint();
            // pb: Java bug 4160279
            // tentative de contournement du bug:
            Component[] comps = CalcoFrame.this.getComponents();
            for (int i=0; i<comps.length; i++)
                if (comps[i] instanceof Canvas)
                    ((Canvas)comps[i]).repaint();
        }
        
        public Dimension getMinimumSize() { return(new Dimension(60, 30)); }
        public Dimension getPreferredSize() { return(getMinimumSize()); }
    }
    
    
    public Double calculer(String s) {
        if ("".equals(s))
            return(null);
        s = ajParentheses(s);
        Parametre param = parser(s);
        Double d = param.evaluer();
        return(d);
    }
    
    public String ajParentheses(String s) {
        String sops = "^/*-+";
        for (int iops=0; iops<sops.length(); iops++) {
            char cops = sops.charAt(iops);
            int indop = s.indexOf(cops);
            if (indop > 0 && "^*/-+E".indexOf(s.charAt(indop-1)) != -1)
                indop = -1; // un - ou + après un E ou un opérateur n'est pas un opérateur !
            int nindop = indop;
            int im,ip;
            char cm=' ',cp=' ';
            int pp;
            boolean ajp;
            while (nindop != -1) {
                ajp = false;
                im = indop - 1;
                if (im >= 0)
                    cm = s.charAt(im);
                pp = 0;
                while (im >= 0 && (pp != 0 || cm != '(') &&
                        (pp != 0 || (sops.indexOf(cm) == -1 ||
                        (im > 0 && "^*/-+E".indexOf(s.charAt(im-1)) != -1)))) {
                    if (cm == ')')
                        pp++;
                    else if (cm == '(')
                        pp--;
                    im--;
                    if (im >= 0)
                        cm = s.charAt(im);
                }
                if (im < 0 || sops.indexOf(cm) != -1)
                    ajp = true;
                ip = indop + 1;
                if (ip >= 0 && ip < s.length())
                    cp = s.charAt(ip);
                pp = 0;
                while (ip < s.length() && (pp != 0 || cp != ')') &&
                        (pp != 0 || (sops.indexOf(cp) == -1 ||
                        (ip > 0 && "^*/-+E".indexOf(s.charAt(ip-1)) != -1)))) {
                    if (cp == '(')
                        pp++;
                    else if (cp == ')')
                        pp--;
                    ip++;
                    if (ip < s.length())
                        cp = s.charAt(ip);
                }
                if (ip >= s.length() || sops.indexOf(cp) != -1)
                    ajp = true;
                if (ajp) {
                    s = s.substring(0, im+1) + "(" + s.substring(im+1, ip) + ")" +
                        s.substring(ip);
                    indop++;
                }
                nindop = s.substring(indop+1).indexOf(cops);
                indop = nindop + indop+1;
            }
        }
        return(s);
    }
    
    public Parametre parser(String s) {
        String sops = "^/*-+";
        if (s.length() > 0 && s.charAt(0) == '(' && s.charAt(s.length()-1) == ')') {
            boolean retparok = true;
            // on vérifie si on peut retirer les parenthèses qui entourent l'expression
            int pp = 0;
            for (int i=1; i<s.length()-1; i++) {
                if (s.charAt(i) == '(')
                    pp++;
                else if (s.charAt(i) == ')')
                    pp--;
                if (pp < 0) {
                    retparok = false;
                    break;
                }
            }
            if (retparok)
                s = s.substring(1, s.length()-1);
        }
        int indop = -1;
        int pp = 0;
        for (int i=0; i<s.length(); i++) {
            if (pp == 0 && sops.indexOf(s.charAt(i)) != -1 &&
                    (i==0 || "^*/-+E".indexOf(s.charAt(i-1)) == -1)) {
                indop = i;
                break;
            } else if (s.charAt(i) == '(')
                pp++;
            else if (s.charAt(i) == ')')
                pp--;
        }
        if (indop == -1) {
            boolean nb = true;
            for (int i=0; i<s.length(); i++)
                if ("0123456789.,E-+ ".indexOf(s.charAt(i)) == -1) {
                    nb = false;
                    break;
                }
            if (nb) {
                double d;
                try {
                    d = (new Double(s)).doubleValue();
                } catch (NumberFormatException ex) {
                    d = 0;
                }
                return(new Nombre(d));
            } else {
                int indf = s.indexOf('(');
                if (indf != -1 && s.charAt(s.length()-1) == ')') {
                    Fonction fct = obtenirFonction(s.substring(0,indf));
                    int indv = 0;
                    Parametre p1 = null;
                    Parametre p2 = null;
                    s = s.substring(indf+1, s.length()-1);
                    indv = s.indexOf(',');
                    if (indv != -1) {
                        p1 = parser(s.substring(0,indv).trim());
                        s = s.substring(indv+1);
                        p2 = parser(s.trim());
                    } else
                        p1 = parser(s.trim());
                    return(new Expression(fct, p1, p2));
                } else
                    return(new Variable(s));
            }
        } else {
            Fonction op = obtenirFonction(""+s.charAt(indop));
            String s1 = s.substring(0,indop).trim();
            if ("".equals(s1))
                s1 = "0"; // -(2+3) = 0 - (2+3)
            Parametre p1 = parser(s1);
            String s2 = s.substring(indop+1).trim();
            Parametre p2 = parser(s2);
            return(new Expression(op, p1, p2));
        }
    }
    
    class Expression implements Parametre {
        Fonction fct;
        Parametre p1,p2;
        
        public Expression(Fonction fct, Parametre p1, Parametre p2) {
            this.fct = fct;
            this.p1 = p1;
            this.p2 = p2;
        }
        public Double evaluer() {
            if (fct == null)
                return(null);
            else if (p2 == null) {
                Double d1 = p1.evaluer();
                if (d1 == null)
                    return(null);
                return(new Double(fct.evaluer(d1.doubleValue(), 0)));
            } else {
                Double d1 = p1.evaluer();
                Double d2 = p2.evaluer();
                if (d1 == null || d2 == null)
                    return(null);
                return(new Double(fct.evaluer(d1.doubleValue(), d2.doubleValue())));
            }
        }
    }
    
    class Variable implements Parametre {
        String nom;
        
        public Variable(String nom) {
            this.nom = nom;
        }
        public Double evaluer() {
            double v;
            Constante cst = mconstantes.getConstante(nom);
            if (cst != null)
                return(new Double(cst.getValeur()));
            return(null);
        }
    }
    
    class Nombre implements Parametre {
        double valeur;
        
        public Nombre(double valeur) {
            this.valeur = valeur;
        }
        public Double evaluer() {
            return(new Double(valeur));
        }
    }
    
    interface Parametre {
        public abstract Double evaluer();
    }
}
